from rest_framework import serializers, fields
from django.conf import settings

from plan.api.exceptions import ValidationError
from plan.api.serializers import (
    CompactServiceSerializer,
    ModelSerializer,
    CompactStaffSerializer,
)
from plan.api.fields import MappingField

from plan.oebs.models import ACTIONS, OEBSAgreement
from plan.oebs.utils import build_oebs_tree
from plan.services.models import Service
from plan.services.state import SERVICE_STATE
from plan.oebs.constants import (
    OEBS_PROCUREMENT_FLAG,
    OEBS_HARDWARE_FLAG,
    OEBS_HR_FLAG,
    OEBS_REVENUE_FLAG,
    STATES,
    OEBS_GROUP_ONLY_FLAG,
)


class OEBSAgreementSerializer(ModelSerializer):
    service = CompactServiceSerializer()
    requester = CompactStaffSerializer()

    class Meta:
        model = OEBSAgreement
        fields = [
            'id', 'created_at', 'updated_at',
            'start_date', 'end_date',
            'issue', 'repair_issue',
            'state', 'service', 'attributes',
            'action', 'move_request_id', 'close_request_id', 'delete_request_id',
            'error_type', 'error_message', 'requester',
        ]


class OEBSAgreementCompactSerializer(ModelSerializer):
    class Meta:
        model = OEBSAgreement
        fields = [
            'issue',
            'action'
        ]


class OEBSCompactServiceSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = Service
        fields = ('id', 'slug', 'name', 'parent_id')


class OEBSAgreementRespondResultSerializer(serializers.Serializer):
    validated = serializers.BooleanField(required=False)
    applied = serializers.BooleanField(required=False)
    leaf_oebs_id = serializers.CharField(required=False)
    group_oebs_id = serializers.CharField(required=False)

    def validate(self, attrs):
        validated = attrs.get('validated', None)
        applied = attrs.get('applied', None)
        if validated is None and applied is None:
            raise ValidationError('`validated` or `applied` required')
        if validated is not None and applied is not None:
            raise ValidationError('`validated` and `applied` are mutual exclusive')
        if applied and not ('leaf_oebs_id' in attrs or 'group_oebs_id' in attrs):
            raise ValidationError('`leaf_oebs_id` and `group_oebs_id` required for applied agreement')
        return attrs


class OEBSAgreementRespondSerializer(serializers.Serializer):
    agreement_id = serializers.IntegerField()
    result = OEBSAgreementRespondResultSerializer()
    error = serializers.JSONField(required=False)

    def validate(self, attrs):
        result = attrs.get('result', {})
        agreement = self._context['agreement']
        if (
            result.get('applied') is not None and agreement.state != STATES.APPLYING_IN_OEBS or
            result.get('validated') is not None and agreement.state != STATES.VALIDATING_IN_OEBS
        ):
            raise ValidationError(f'Inconsistent agreement state {agreement.state}')
        return attrs


class OEBSTreeServiceSerializer(ModelSerializer):
    agreement = OEBSAgreementCompactSerializer()
    group_oebs_id = serializers.SerializerMethodField()
    leaf_oebs_id = serializers.SerializerMethodField()

    class Meta:
        model = Service
        fields = [
            'id',
            'slug',
            'name',
            'name_en',
            'state',
            'use_for_hardware',
            'use_for_hr',
            'use_for_procurement',
            'use_for_revenue',
            'use_for_group_only',
            'agreement',
            'group_oebs_id',
            'leaf_oebs_id',
        ]

    def get_fields(self):
        fields = super(OEBSTreeServiceSerializer, self).get_fields()
        fields['child_nodes'] = OEBSTreeServiceSerializer(many=True)
        return fields

    def get_group_oebs_id(self, instance):
        return instance.oebs_product.get('parent_oebs_id') if instance.oebs_product else None

    def get_leaf_oebs_id(self, instance):
        return instance.oebs_product.get('leaf_oebs_id') if instance.oebs_product else None


class OEBSTreeAgreementSerializer(ModelSerializer):
    class Meta:
        model = OEBSAgreement
        fields = ['id']

    def recursive_state_replacement(self, service_node, state):
        service_node['state'] = state
        for child in service_node['child_nodes']:
            self.recursive_state_replacement(child, state)

    def data_replacement(self, service_node, action, flags):
        states_map = {
            ACTIONS.CLOSE: SERVICE_STATE.CLOSED,
            ACTIONS.DELETE: SERVICE_STATE.DELETED,
        }

        if action in [ACTIONS.CLOSE, ACTIONS.DELETE]:
            self.recursive_state_replacement(service_node, states_map[action])

        elif action == ACTIONS.CHANGE_FLAGS:
            for flag, value in flags.items():
                service_node[flag] = value

    def search_service_node(self, service_node, service_id):
        if service_node['id'] == service_id:
            return service_node

        for child in service_node['child_nodes']:
            result = self.search_service_node(child, service_id)

            if result is not None:
                return result

    def to_representation(self, instance):
        tree = build_oebs_tree(instance)
        result = OEBSTreeServiceSerializer(instance=tree).data

        service_id = instance.service_id
        if instance.action != ACTIONS.MOVE:
            # в build_oebs_tree уже отстроили новое дерево,
            # поэтому для MOVE подменять ничего не надо
            flags = instance.get_target_flags()
            service_node = self.search_service_node(result, service_id)
            self.data_replacement(service_node, instance.action, flags)

        return result


class OEBSServiceSerializer(ModelSerializer):
    group_oebs_id = serializers.SerializerMethodField()
    leaf_oebs_id = serializers.SerializerMethodField()
    is_vs = serializers.SerializerMethodField()
    is_bu = serializers.SerializerMethodField()
    is_exp = serializers.SerializerMethodField()
    parent_id = serializers.SerializerMethodField()
    name = serializers.SerializerMethodField()
    name_en = serializers.SerializerMethodField()

    class Meta:
        model = Service
        fields = (
            'id', 'slug', 'name',
            'name_en', 'use_for_hardware', 'use_for_hr',
            'use_for_procurement', 'use_for_revenue',
            'use_for_group_only',
            'group_oebs_id', 'leaf_oebs_id',
            'parent_id', 'is_vs', 'is_bu', 'is_exp',
        )

    def get_name(self, instance):
        if instance.oebs_agreement.action == ACTIONS.RENAME:
            return instance.oebs_agreement.attributes['new_ru']
        return instance.name

    def get_name_en(self, instance):
        if instance.oebs_agreement.action == ACTIONS.RENAME:
            return instance.oebs_agreement.attributes['new_en']
        return instance.name_en

    def get_group_oebs_id(self, instance):
        return instance.oebs_product.get('parent_oebs_id')

    def get_leaf_oebs_id(self, instance):
        return instance.oebs_product.get('leaf_oebs_id')

    def get_is_vs(self, instance):
        return settings.GRADIENT_VS in instance.tag_slugs

    def get_is_bu(self, instance):
        return settings.BUSINESS_UNIT_TAG in instance.tag_slugs

    def get_is_exp(self, instance):
        if instance.parent:
            return instance.parent.slug == settings.OEBS_EXPERIMENTS_ROOT
        return False

    def get_parent_id(self, instance):
        return instance.oebs_parent_id


class OEBSActiveAgreementSerializer(ModelSerializer):
    agreement_id = serializers.IntegerField(source='id')
    service = OEBSServiceSerializer()
    use_for_hardware = serializers.SerializerMethodField()
    use_for_hr = serializers.SerializerMethodField()
    use_for_procurement = serializers.SerializerMethodField()
    use_for_revenue = serializers.SerializerMethodField()
    use_for_group_only = serializers.SerializerMethodField()
    parent_id = serializers.SerializerMethodField()

    def get_use_for_hardware(self, instance):
        return instance.attributes.get(OEBS_HARDWARE_FLAG, instance.service.use_for_hardware)

    def get_use_for_group_only(self, instance):
        return instance.attributes.get(OEBS_GROUP_ONLY_FLAG, instance.service.use_for_group_only)

    def get_use_for_hr(self, instance):
        return instance.attributes.get(OEBS_HR_FLAG, instance.service.use_for_hr)

    def get_use_for_procurement(self, instance):
        return instance.attributes.get(OEBS_PROCUREMENT_FLAG, instance.service.use_for_procurement)

    def get_use_for_revenue(self, instance):
        return instance.attributes.get(OEBS_REVENUE_FLAG, instance.service.use_for_revenue)

    def get_parent_id(self, instance):
        return instance.oebs_parent_id

    class Meta:
        model = OEBSAgreement
        fields = (
            'agreement_id', 'action', 'service',
            'use_for_hardware', 'use_for_hr',
            'use_for_procurement', 'use_for_revenue',
            'parent_id', 'use_for_group_only',
        )


class OEBSDeviationSerializer(ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    parent = OEBSCompactServiceSerializer()
    valuestream = OEBSCompactServiceSerializer()
    abc_oebs_parent_id = serializers.IntegerField()
    group_oebs_id = serializers.SerializerMethodField()
    leaf_oebs_id = serializers.SerializerMethodField()
    oebs_parent = serializers.SerializerMethodField()
    oebs_data = fields.JSONField(allow_null=True)
    fields_mapping_ = {'name': ('name', 'name_en')}

    def get_group_oebs_id(self, instance):
        return instance.oebs_product.get('parent_oebs_id')

    def get_leaf_oebs_id(self, instance):
        return instance.oebs_product.get('leaf_oebs_id')

    def get_oebs_parent(self, instance):
        if instance.oebs_parent:
            return OEBSCompactServiceSerializer(instance.oebs_parent).data

    class Meta:
        model = Service
        fields = (
            'id', 'slug', 'name', 'leaf_oebs_id', 'group_oebs_id',
            'oebs_parent_id', 'oebs_parent',
            'oebs_data', 'abc_oebs_parent_id',
            'parent', 'valuestream', 'use_for_hr', 'use_for_revenue',
            'use_for_procurement', 'use_for_group_only',
        )
