from django.conf import settings
from django.shortcuts import get_object_or_404
from rest_framework import viewsets, mixins
from rest_framework.response import Response

from plan.api.base import ABCCursorPagination
from plan.api.exceptions import BadRequest
from plan.api.filters import PlanFilterSet, IntegerListFilter
from plan.api.mixins import DefaultFieldsMixin, OrderingMixin, SelectOnlyFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.oebs.api.permissions import OEBSAuthenticated
from plan.oebs.api.serializers import (
    OEBSAgreementSerializer,
    OEBSTreeAgreementSerializer,
    OEBSActiveAgreementSerializer,
    OEBSAgreementRespondSerializer,
    OEBSDeviationSerializer,
)
from plan.oebs.models import OEBSAgreement
from plan.oebs.tasks.start_approve import start_oebs_approve_process
from plan.oebs.tasks.finish_approve import finish_oebs_approve_process
from plan.oebs.constants import STATES, ACTIONS
from plan.oebs.utils import (
    fail_service_request,
    get_oebs_products,
    get_agreement_tags,
    get_parents_map,
    handle_oebs_error,
    _build_parents,
)
from plan.swagger import SwaggerFrontend
from plan.services.models import Service


class OEBSAgreementFilterSet(PlanFilterSet):
    service__in = IntegerListFilter(field_name='service', lookup_expr='in')

    class Meta:
        model = OEBSAgreement
        fields = {
            'state': ['exact', 'in'],
            'service': ['exact'],
            'start_date': ['gte', 'lte', 'range'],
            'end_date': ['gte', 'lte', 'range'],
        }


class OEBSAgreementView(DefaultFieldsMixin,
                        SelectOnlyFieldsMixin,
                        TvmAccessMixin,
                        viewsets.ReadOnlyModelViewSet):
    default_swagger_schema = SwaggerFrontend

    queryset = OEBSAgreement.objects.all()
    serializer_class = OEBSAgreementSerializer
    filterset_class = OEBSAgreementFilterSet
    pagination_class = ABCCursorPagination
    ordering_fields = ('id',)
    ordering = ('-id',)
    default_cursor_ordering = ('-id',)
    default_fields = [
        'id', 'start_date', 'end_date',
        'issue', 'state', 'action', 'requester',
    ]


class OEBSTreeView(TvmAccessMixin, viewsets.GenericViewSet, mixins.RetrieveModelMixin):
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'view_all_services'
    permission_classes = [TvmAuthenticated]
    serializer_class = OEBSTreeAgreementSerializer
    queryset = OEBSAgreement.objects.active().select_related(
        'move_request',
        'move_request__destination',
        'service',
    )


class V4OEBSAgreementView(OrderingMixin, TvmAccessMixin, mixins.ListModelMixin, viewsets.GenericViewSet):
    _permissions_to_proceed = 'view_all_services'
    permission_classes = [TvmAuthenticated]
    pagination_class = ABCCursorPagination
    serializer_class = OEBSActiveAgreementSerializer
    ordering_fields = ('id',)
    queryset = OEBSAgreement.objects.active().select_related(
        'move_request',
        'move_request__destination',
        'service',
        'service__parent',
    )

    def get_serializer(self, queryset, *args, **kwargs):
        services_ids = [agreement.service_id for agreement in queryset]
        service_parents_map = get_parents_map(queryset)
        oebs_products = get_oebs_products(
            services_ids
            + list(service_parents_map.values())
            + list(service_parents_map.keys())
        )
        tag_slugs = get_agreement_tags(services_ids)

        for agreement in queryset:
            setattr(agreement.service, 'oebs_product', oebs_products.get(agreement.service_id, {}))
            setattr(agreement.service, 'tag_slugs', tag_slugs.get(agreement.service_id, set()))
            setattr(agreement.service, 'oebs_agreement', agreement)

            parent_id = agreement.service.parent_id
            if parent_id is None:
                parent_id = 0
            else:
                while parent_id is not None and parent_id not in oebs_products:
                    parent_id = service_parents_map.get(parent_id)
            agreement.oebs_parent_id = parent_id
            agreement.service.oebs_parent_id = parent_id

            if agreement.action == ACTIONS.MOVE:
                parent_id = agreement.move_request.destination_id
                if parent_id is None:
                    # двигают в корень
                    parent_id = 0
                else:
                    while parent_id is not None and parent_id not in oebs_products:
                        parent_id = service_parents_map.get(parent_id)
                agreement.oebs_parent_id = parent_id

        return super().get_serializer(queryset, *args, **kwargs)

    def filter_queryset(self, queryset):
        action = self.request.query_params.get('action')
        if not action:
            raise BadRequest('No action specified')
        if action == 'validate':
            queryset = queryset.filter(state=STATES.VALIDATING_IN_OEBS)
        elif action == 'apply':
            queryset = queryset.filter(state=STATES.APPLYING_IN_OEBS)
        else:
            raise BadRequest('Incorrect action specified')

        return queryset


class OEBSRespondView(TvmAccessMixin, viewsets.ViewSet):
    TVM_ALLOWED_METHODS = {'POST'}
    permission_classes = [OEBSAuthenticated]

    def create(self, request):
        agreement = get_object_or_404(OEBSAgreement, pk=request.data.get('agreement_id'))
        serializer = OEBSAgreementRespondSerializer(data=request.data, context={'agreement': agreement})
        serializer.is_valid(raise_exception=True)
        data = serializer.data
        result = data['result']
        errors = data.get('error')
        if result.get('applied'):
            agreement.applied_in_oebs()
            finish_oebs_approve_process.apply_async(
                kwargs={
                    'agreement_id': agreement.id,
                    'leaf_oebs_id': result['leaf_oebs_id'],
                    'parent_oebs_id': result['group_oebs_id'],
                },
                countdown=settings.ABC_DEFAULT_COUNTDOWN,
            )
        elif result.get('validated'):
            agreement.validated()
            start_oebs_approve_process.apply_async(args=[agreement.id], countdown=settings.ABC_DEFAULT_COUNTDOWN)
        else:
            handle_oebs_error(agreement, errors)
            if agreement.action in [ACTIONS.MOVE, ACTIONS.CLOSE, ACTIONS.DELETE]:
                fail_service_request(agreement)
        return Response(status=200)


class OEBSDeviationView(OrderingMixin, TvmAccessMixin, mixins.ListModelMixin, viewsets.GenericViewSet):
    _permissions_to_proceed = 'view_all_services'
    permission_classes = [TvmAuthenticated]
    pagination_class = ABCCursorPagination
    serializer_class = OEBSDeviationSerializer
    ordering_fields = ('id', )

    def get_queryset(self):
        return Service.objects.oebs_deviation().select_related(
            'valuestream', 'parent',
        )

    def get_serializer(self, queryset, *args, **kwargs):
        services_ids = [service.id for service in queryset]
        oebs_parents = {
            parent.id : parent
            for parent in Service.objects.filter(
                pk__in=(service.oebs_parent_id for service in queryset)
            ).select_related('parent')
        }

        parents_map = {}
        for service in queryset:
            parents_map.update(dict(_build_parents(service)))

        oebs_products = get_oebs_products(
            services_ids
            + list(parents_map.values())
            + list(parents_map.keys())
        )

        for service in queryset:
            setattr(service, 'oebs_product', oebs_products.get(service.id, {}))
            setattr(service, 'oebs_parent', oebs_parents.get(service.oebs_parent_id))
            parent_id = service.parent_id
            if parent_id is None:
                parent_id = 0
            else:
                while parent_id is not None and parent_id not in oebs_products:
                    parent_id = parents_map.get(parent_id)
            service.abc_oebs_parent_id = parent_id

        return super().get_serializer(queryset, *args, **kwargs)
