import logging
from typing import Optional

from django.contrib.postgres.fields import JSONField
from django.db import models
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _
from django.conf import settings

from plan.common.models import TimestampedModel
from plan.common.utils.startrek import create_comment, change_state, close_issue

from plan.common.utils.enum import EnumChoice
from .utils import is_oebs_related
from .constants import (
    OEBS_FLAGS,
    ACTIONS,
    STATES,
    ERRORS,
)

log = logging.getLogger(__name__)


class OEBSAgreementQuerySet(models.QuerySet):

    def active(self):
        return self.exclude(state__in=STATES.FINAL_STATES)

    def not_applied(self):
        return self.exclude(state=STATES.APPLIED)


class OEBSAgreementManager(models.Manager.from_queryset(OEBSAgreementQuerySet)):

    def rename_service(self, service, requester, name, name_en, old_name, old_name_en) -> None:
        return OEBSAgreement.objects.create(
            service=service,
            action=ACTIONS.RENAME,
            state=STATES.VALIDATED_IN_OEBS,
            requester=requester,
            attributes={
                'ru': old_name,
                'en': old_name_en,
                'new_ru': name,
                'new_en': name_en,
            },
        )

    def change_flags(self, service, requester, updated_flags):
        return self.create(
            service=service,
            attributes=updated_flags,
            action=ACTIONS.CHANGE_FLAGS,
            requester=requester,
            state=STATES.VALIDATING_IN_OEBS,
        )

    def move_service_create(self, service, requester, move_request):
        notify_only = not is_oebs_related(service, ignore_dormant=True)
        self.create(
            service=service,
            action=ACTIONS.MOVE,
            notify_only=notify_only,
            requester=requester,
            move_request=move_request,
        )

    def close_service(self, service, requester, close_request):
        self.create(
            service=service,
            action=ACTIONS.CLOSE,
            requester=requester,
            close_request=close_request,
            state=STATES.VALIDATING_IN_OEBS,
            attributes={
                flag: False
                for flag in OEBS_FLAGS
            }
        )

        close_request.start_oebs_process()
        close_request.save()

    def delete_service(self, service, requester, delete_request):
        self.create(
            service=service,
            action=ACTIONS.DELETE,
            requester=requester,
            state=STATES.VALIDATING_IN_OEBS,
            delete_request=delete_request,
            attributes={
                flag: False
                for flag in OEBS_FLAGS
            }
        )
        delete_request.start_oebs_process()
        delete_request.save()


class OEBSAgreement(TimestampedModel):
    ok_id = models.CharField(max_length=50, blank=True, null=True, verbose_name=_('Id согласования в OK'))
    issue = models.CharField(max_length=32, blank=True, null=True, verbose_name=_('Тикет на согласование'))
    repair_issue = models.CharField(max_length=32, blank=True, null=True, verbose_name=_('Тикет на починку'))
    start_date = models.DateField(blank=True, null=True, verbose_name=_('Начало согласования'))
    end_date = models.DateField(blank=True, null=True, verbose_name=_('Окончание согласования'))
    notify_only = models.BooleanField(blank=True, default=False, verbose_name=_('Не требует согласования'))
    state = models.CharField(
        max_length=32, default=STATES.OPEN, verbose_name=_('Статус'),
        choices=[
            (state, STATES[state].name) for state in STATES
        ]
    )
    requester = models.ForeignKey(
        'staff.Staff',
        related_name='oebs_agreements',
        verbose_name=_('Запросивший'),
        null=True, blank=True,
    )

    service = models.ForeignKey(
        'services.Service',
        on_delete=models.CASCADE,
        related_name='oebs_agreements',
    )
    attributes = JSONField(default=dict, verbose_name=_('Атрибуты'))
    action = models.CharField(
        max_length=32, verbose_name='Тип согласования',
        choices=[
            (action, ACTIONS[action].name) for action in ACTIONS
        ],
    )
    move_request = models.ForeignKey(
        'services.ServiceMoveRequest',
        blank=True, null=True, on_delete=models.SET_NULL,
        verbose_name=_('Запрос на перемещение'),
    )
    close_request = models.ForeignKey(
        'services.ServiceCloseRequest',
        blank=True, null=True, on_delete=models.SET_NULL,
        verbose_name=_('Запрос на закрытие'),
    )
    delete_request = models.ForeignKey(
        'services.ServiceDeleteRequest',
        blank=True, null=True, on_delete=models.SET_NULL,
        verbose_name=_('Запрос на удаление'),
    )
    error_type = models.CharField(
        max_length=32, verbose_name=_('Тип ошибки'),
        blank=True, null=True,
        choices=[
            (error, ERRORS[error].name) for error in ERRORS
        ]
    )
    error_message = JSONField(blank=True, null=True, verbose_name=_('Ошибка'))

    objects = OEBSAgreementManager()

    def get_target_flags(self) -> dict:
        return {
            flag: self.attributes.get(flag, getattr(self.service, flag))
            for flag in OEBS_FLAGS
        }

    def start(self):
        logging.info(f'Start OEBSAgreement approval - {self.pk}')
        self.state = STATES.APPROVING
        self.start_date = timezone.now().date()
        self.save(update_fields=['state', 'start_date'])

    def approve(self):
        logging.info(f'OEBSAgreement approved - {self.pk}')
        self.state = STATES.APPROVED
        self.save(update_fields=['state'])

    def validated(self):
        logging.info(f'OEBSAgreement was validated - {self.pk}')
        self.state = STATES.VALIDATED_IN_OEBS
        self.save(update_fields=['state'])

    def applying_in_oebs(self):
        logging.info(f'Waiting for OEBSAgreement applying in OEBS - {self.pk}')
        self.state = STATES.APPLYING_IN_OEBS
        self.save(update_fields=['state'])

    def applied_in_oebs(self):
        logging.info(f'OEBSAgreement applied in OEBS - {self.pk}')
        self.state = STATES.APPLIED_IN_OEBS
        self.save(update_fields=['state'])

    def fail(self, error: EnumChoice, message: Optional[dict] = None, repair_issue: Optional[str] = None) -> None:
        logging.info(f'Marking OEBSAgreement as failed - {self.pk}')
        self.state = STATES.FAILED
        self.end_date = timezone.now().date()
        self.error_type = error
        self.error_message = message
        self.repair_issue = repair_issue
        self.save(update_fields=['state', 'end_date', 'error_type', 'error_message', 'repair_issue'])
        if self.issue:
            if repair_issue:
                comment = f'Произошла ошибка при применении изменений, исправление будет произведено в рамках - {repair_issue}'
            else:
                comment = 'Произошла ошибка при применении изменений'
            close_issue(self.issue, comment=comment)

    def decline(self, fail_request=True):
        from plan.oebs.utils import fail_service_request

        logging.info(f'Marking OEBSAgreement as declined - {self.pk}')
        self.state = STATES.DECLINED
        self.end_date = timezone.now().date()
        self.save(update_fields=['state', 'end_date'])
        if self.issue:
            close_issue(
                self.issue,
                comment='Согласование отклонено, изменения не применены',
            )
        if fail_request:
            fail_service_request(self)

    def start_moving_service(self):
        self.move_request.start_oebs_process()
        self.move_request.save()
        self.state = STATES.VALIDATING_IN_OEBS
        self.save(update_fields=('state', ))

    def get_new_names(self):
        return self.attributes['new_ru'], self.attributes['new_en']

    def apply(self, leaf_oebs_id, parent_oebs_id):
        self.state = STATES.APPLIED
        self.end_date = timezone.now().date()
        if self.issue:
            create_comment(
                key=self.issue,
                comment_text=(
                    'Процесс применения изменений успешно завершен.\n'
                    f'**ID листового узла в OEBS:** %%{leaf_oebs_id}%%\n'
                    f'**ID родительского узла в OEBS:** %%{parent_oebs_id}%%'
                ),
            )
            change_state(key=self.issue, transition=settings.OEBS_TICKET_IN_PROGRESS)

        self.save(update_fields=['state', 'end_date', ])

    def __str__(self):
        return (f'OEBSAgreement {self.pk}, '
                f'service: {self.service_id}, '
                f'action: {self.action}, '
                f'state: {self.state}')
