import logging

from plan.common.utils.tasks import lock_task
from plan.common.utils.ok import OkClient

from plan.services.models import Service, ServiceTag
from ..models import (
    OEBSAgreement,
    STATES,
    ACTIONS,
    ERRORS,
    OEBS_FLAGS,
)
from ..utils import oebs_approval_time_is_over
from .start_approve import start_oebs_approve_process

log = logging.getLogger(__name__)


@lock_task()
def check_oebs_approve_process():
    """
    Регулярная таска для обхода все активных OEBSAgreement и проверки их состояния
    """
    agreements_to_check = OEBSAgreement.objects.active()
    ok_client = OkClient()
    for agreement in agreements_to_check:
        if agreement.action == ACTIONS.MOVE and not agreement.move_request.approved:
            continue

        if agreement.state in (STATES.VALIDATING_IN_OEBS, STATES.APPLYING_IN_OEBS):
            # ждем пока на стороне OEBS будут провалидированы/применены изменения, ничего не делаем
            continue

        if agreement.state == STATES.APPLIED_IN_OEBS:
            # по-хорошему тут нужно запустить finish_oebs_approve_process, но у нас нет leaf/group id
            # начать сохранять в базу, если будут с этим проблемы
            log.error(f'Found stale applied in OEBS OEBSAgreement - {agreement.id}')
            continue

        if not agreement.issue:
            if agreement.state != STATES.VALIDATED_IN_OEBS:
                logging.error(f'Found active OEBSAgreement without issue - {agreement.id}')
            else:
                start_oebs_approve_process.delay(agreement.id)
            continue

        ok_id = agreement.ok_id
        if not ok_id:
            logging.error(f'Found active OEBSAgreement without ok_id - {agreement.id}')
            continue

        ok_status, ok_resolution = ok_client.get_request_state(agreement)

        if ok_status == 'closed':
            if ok_resolution == 'approved':
                agreement.applying_in_oebs()
            elif ok_resolution == 'declined':
                agreement.decline()
        elif ok_status == 'rejected':
            agreement.decline()
        elif ok_status == 'in_progress' and oebs_approval_time_is_over(agreement):
            agreement.fail(
                error=ERRORS.NOT_APPROVED_IN_TIME,
            )


@lock_task()
def sync_tags(root_id: int = None, dry_run: bool = False) -> None:
    """
    Проверят и если нужно правит выставленные OEBS теги в соответствии
    с галочками в базе
    :param root_id: id сервиса с которого начинать обход вниз
    :param dry_run: применять реально изменения или нет
    """

    if root_id:
        services = Service.objects.get(pk=root_id).get_descendants(include_self=True)
    else:
        services = Service.objects.all()

    for service in services:
        to_add = []
        to_del = []
        current_tags = {tag.slug for tag in service.tags.all()}
        for flag in OEBS_FLAGS:
            current_value = getattr(service, flag)
            tag_slug = OEBS_FLAGS[flag]
            if current_value != (tag_slug in current_tags):
                if current_value:
                    to_add.append(tag_slug)
                else:
                    to_del.append(tag_slug)

        if to_add:
            if not dry_run:
                service.tags.add(*ServiceTag.objects.filter(slug__in=to_add))
            log.info(f'Add tags {to_add} to {service.slug} - {dry_run}')
        if to_del:
            if not dry_run:
                service.tags.remove(*ServiceTag.objects.filter(slug__in=to_del))
            log.info(f'Remove tags {to_del} from {service.slug} - {dry_run}')
