import logging

from plan.common.utils.tasks import lock_task
from plan.oebs.models import OEBSAgreement
from plan.oebs.utils import (
    create_oebs_resource,
    finalize_service_request,
    finalize_flag_changing,
    is_oebs_related
)
from plan.oebs.constants import ACTIONS
from plan.oebs.constants import STATES

log = logging.getLogger(__name__)


@lock_task(lock_key=lambda agreement_id, *args, **kwargs: 'finish_oebs_approve_process_{}'.format(agreement_id))
def finish_oebs_approve_process(agreement_id: int, leaf_oebs_id: str, parent_oebs_id: str):
    """
    Выполняет действия после согласования OEBSAgreement на стороне Трекера
    и после применения изменений на стороне OEBS
    """

    agreement = OEBSAgreement.objects.get(pk=agreement_id)
    if agreement.state != STATES.APPLIED_IN_OEBS:
        log.error('Can finish only agreements with APPLIED_IN_OEBS state')
        return

    service = agreement.service
    if agreement.action == ACTIONS.CHANGE_FLAGS and not is_oebs_related(service, with_descendants=False):
        # создать ресурс
        create_oebs_resource(
            oebs_agreement=agreement,
            leaf_oebs_id=leaf_oebs_id,
            parent_oebs_id=parent_oebs_id,
        )

    finalize_flag_changing(agreement)

    if agreement.action in [ACTIONS.MOVE, ACTIONS.CLOSE, ACTIONS.DELETE, ACTIONS.RENAME]:
        finalize_service_request(agreement)

    agreement.apply(leaf_oebs_id=leaf_oebs_id, parent_oebs_id=parent_oebs_id)
