import logging

from ylog.context import log_context

from plan.common.utils.tasks import lock_task
from plan.common.utils.ok import OkClient
from plan.oebs.utils import (
    get_oebs_approvers,
    create_oebs_approve_issue,
    add_money_map_data,
)
from plan.oebs.constants import STATES

log = logging.getLogger(__name__)


@lock_task(lock_key=lambda agreement_id, *args, **kwargs: 'start_oebs_approve_process_{}'.format(agreement_id))
def start_oebs_approve_process(agreement_id: int):
    """
    Запускает процесс согласования связанных с OEBS изменений
    """
    from plan.oebs.models import OEBSAgreement
    with log_context(oebs_agreement=agreement_id):
        log.info('Starting approving process for OEBSAgreement')
        agreement = OEBSAgreement.objects.get(pk=agreement_id)
        if agreement.state != STATES.VALIDATED_IN_OEBS:
            log.error('Can start only agreements with VALIDATED_IN_OEBS state')
            return

        ok_client = OkClient()

        if agreement.notify_only:
            log.info('Skip approval for OEBSAgreement')
            agreement.applying_in_oebs()
        else:
            approvers = get_oebs_approvers(agreement)
            if not approvers:
                # не удалось вычислить ответственных за согласование
                # agreement был помечен как fail
                return

            create_oebs_approve_issue(agreement)
            log.info('Create issue for OEBSAgreement')
            ok_client.create_request(agreement, approvers)
            log.info('Start OK process for OEBSAgreement')
            ok_client.add_request_to_ticket(agreement)
            log.info('Create OK comment for OEBSAgreement')
            add_money_map_data(agreement)
            agreement.start()
            log.info(f'OEBSAgreement now approving, issue: {agreement.issue}, ok_id: {agreement.ok_id}')
