import json
import logging
from collections import defaultdict

from datetime import timedelta
from typing import Iterable, Optional

from django.conf import settings
from django.db.models import Q
from django.template.defaulttags import register
from django.template.loader import render_to_string
from django.utils.translation import ugettext as _
from ids.exceptions import IDSException, BackendError

from plan.api.exceptions import FailedDependency
from plan.common.utils import timezone
from plan.common.utils.startrek import (
    create_issue,
    change_state,
    update_issue,
    create_comment,
)
from plan.roles.models import Role
from plan.oebs.constants import (
    ACTIONS,
    ERRORS,
    OEBS_FLAGS,
    OEBS_HARDWARE_FLAG,
    OEBS_HR_FLAG,
    OEBS_PROCUREMENT_FLAG,
    OEBS_REVENUE_FLAG,
    OEBS_MATCHING_FLAGS_AND_ROLES,
    OEBS_GROUP_ONLY_FLAG,
    STATES,
)
from plan.oebs.exceptions import NoAncestors

log = logging.getLogger(__name__)

MONEY_MAP_COMMENT = '''Обращаем также внимание, что к сервису привязаны рекламные площадки, их можно посмотреть тут: {}
В случае переименования сервиса они все автоматически будут продолжать быть к нему привязанными,
только в отчетностях будет другое название
'''


def is_oebs_related(service: 'Service', with_descendants: bool = True, ignore_dormant: bool = False) -> bool:
    """
    Проверяет является ли указанный сервис (или любой из его дочерних) связанным с oebs.

    Задав параметр ignore_dormant=True, можно исключить при формировании ответа "бывшие"
    OEBS-связанные сервисы, у которых есть OEBS ресурс, но нет включеных OEBS флагов.
    """
    from plan.resources.models import ServiceResource   # тк импорты циклятся

    services = [service, ]
    if with_descendants:
        services = service.get_alive_descendants(include_self=True)
        if ignore_dormant:
            services = services.with_oebs_flags()
    elif ignore_dormant and not service.with_oebs_flags():
        return False

    return ServiceResource.objects.filter(
        service__in=services,
        type__code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE,
    ).exists()


def finalize_service_request(oebs_agreement):
    from plan.services.tasks import delete_service, close_service, move_service, rename_service

    action_map = {
        ACTIONS.RENAME: (rename_service, oebs_agreement.service_id),
        ACTIONS.MOVE: (move_service, oebs_agreement.move_request_id),
        ACTIONS.DELETE: (delete_service, oebs_agreement.delete_request_id),
        ACTIONS.CLOSE: (close_service, oebs_agreement.close_request_id),
    }
    task, request_id = action_map.get(oebs_agreement.action, (None, None))
    if task:
        task.apply_async(
            args=[request_id],
            countdown=settings.ABC_DEFAULT_COUNTDOWN,
        )
    if oebs_agreement.action == ACTIONS.RENAME:
        # сменим ответственного, чтобы коллеги переименовали в hype
        name, name_en = oebs_agreement.get_new_names()
        oebs_agreement.service.name = name
        oebs_agreement.service.name_en = name_en
        oebs_agreement.service.save(update_fields=('name', 'name_en',))
        update_issue(
            key=oebs_agreement.issue,
            assignee=get_assignee_for_renaming(),
        )


def fail_service_request(oebs_agreement):
    """
    Переводит запрос на перемещение/удаление/закрытие в статус Ошибка.
    """
    from plan.services.models import ServiceMoveRequest, ServiceDeleteRequest, ServiceCloseRequest

    request_map = {
        ACTIONS.MOVE: (ServiceMoveRequest, oebs_agreement.move_request_id),
        ACTIONS.DELETE: (ServiceDeleteRequest, oebs_agreement.delete_request_id),
        ACTIONS.CLOSE: (ServiceCloseRequest, oebs_agreement.close_request_id),
    }
    model, request_id = request_map.get(oebs_agreement.action, (None, None))

    if model:
        request = model.objects.get(pk=request_id)
        request.fail()
        request.save()


def notify_agreement_staff_not_found(agreement, role: Role):
    error_message = _(f'В вышестоящих сервисах отсутствует сотрудник с ролью {role.name}')

    context = {
        'error_message': error_message,
        'service': agreement.service
    }

    issue_key = None
    try:
        issue = create_issue(
            queue=settings.OEBS_AGREEMENT_ERROR_QUEUE,
            summary='Не найден сотрудник для подтверждения изменений в ГК OEBS',
            description=render_to_string('oebs_agreement_error_issue.txt', context),
            tags=settings.OEBS_AGREEMENT_ERROR_ISSUE_TAGS,
            components=settings.OEBS_AGREEMENT_ERROR_ISSUE_COMPONENTS,
            createdBy=agreement.requester.login,
            assignee=settings.OEBS_AGREEMENT_ERROR_ISSUE_ASSIGNEE
        )
        issue_key = issue.key
        issue.comments.create(summonees=[agreement.requester.login])
    except IDSException as exc:
        log.warning(f'Failed to create repair issue for {agreement}: {exc}')
    log.warning(f'Mark OEBSAgreement {agreement.pk} as failed')
    agreement.fail(ERRORS.NO_APPROVERS, {'message': error_message}, issue_key)


def handle_oebs_error(agreement, error_message: dict) -> None:
    service = agreement.service
    context = {
        'requester': agreement.requester,
        'service': service,
        'agreement': agreement,
        'error': json.dumps(error_message),
    }
    if agreement.repair_issue:
        repair_issue_key = agreement.repair_issue
    else:
        try:
            repair_issue = create_issue(
                settings.OEBS_SUPPORT_QUEUE,
                assignee=settings.OEBS_SUPPORT_ASSIGNEE,
                summary=f'Ошибка при изменении сервиса {service.name}',
                description=render_to_string('oebs_support_error_ticket.txt', context),
                tags=settings.OEBS_AGREEMENT_ERROR_ISSUE_TAGS,
            )
        except BackendError as exc:
            repair_issue_key = None
            log.error(f'Failed to create OEBS repair issue for {agreement}: {exc}')
        else:
            repair_issue_key = repair_issue.key

    agreement.fail(ERRORS.OEBS_ERROR, message=error_message, repair_issue=repair_issue_key)


def oebs_approval_time_is_over(oebs_agreement) -> bool:
    start_date = oebs_agreement.start_date
    if start_date + timedelta(days=30) < timezone.now().date():
        return True
    return False


def build_oebs_subtree(service_pk, tree_services, services_to_children, oebs_products, agreement):
    service = tree_services[service_pk]

    service.oebs_product = oebs_products.get(service_pk)
    has_oebs_in_subtree = (service.oebs_product is not None)

    if service_pk == agreement.service.pk:
        service.agreement = agreement
        has_oebs_in_subtree = True
    else:
        service.agreement = None

    service.child_nodes = []
    for child in services_to_children[service_pk]:
        child_subtree, child_has_oebs = build_oebs_subtree(
            child, tree_services, services_to_children, oebs_products, agreement
        )
        if child_has_oebs:
            service.child_nodes.append(child_subtree)
            has_oebs_in_subtree = True
    return service, has_oebs_in_subtree


def get_oebs_products(services_ids):
    from plan.resources.models import ServiceResource

    return {
        sid: attributes for sid, attributes in
            ServiceResource.objects.filter(
            resource__type__code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE,
            service_id__in=services_ids,
        ).values_list('service_id', 'attributes')
    }


def get_parents_map(agreements):
    parents_map = {}
    for agreement in agreements:
        parents_map.update(dict(_build_parents(agreement.service)))
        if agreement.action == ACTIONS.MOVE:
            if agreement.move_request.destination:
                parents_map.update(dict(_build_parents(agreement.move_request.destination)))
    return parents_map


def _build_parents(service):
    if not service.ancestors and service.parent_id is not None:
        raise NoAncestors(f'Service {service.id} has no ancestors')
    else:
        if not service.ancestors:
            yield service.id, None
        else:
            service_id = service.id
            for ancestor in service.ancestors[::-1]:
                yield service_id, ancestor['id']
                service_id = ancestor['id']


def get_agreement_tags(services_ids):
    from plan.services.models import ServiceTag
    result = defaultdict(set)

    tags_data = ServiceTag.objects.filter(
        service__id__in=services_ids,
        slug__in=[settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG],
    ).values_list('slug', 'service')

    for slug, service in tags_data:
        result[service].add(slug)

    return result


def build_oebs_tree(agreement):
    from plan.services.models import Service

    if agreement.action == ACTIONS.MOVE:
        origin_service = agreement.move_request.destination
    else:
        origin_service = agreement.service

    root_service = (
        origin_service
        .get_ancestors(include_self=True)
        .filter(tags__slug__in=[settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG])
        .order_by('-level')
        .first()
    )

    if root_service is None:
        raise FailedDependency(
            message={
                'ru': "Выше по дереву нет тега 'vs' или 'bu'.",
                'en': "No ancestors has the tag 'vs' or 'bu'.",
            }
        )

    tree_services_pks = root_service.get_descendants(include_self=True).values_list('pk', flat=True)
    move_tree_services_pks = []
    if agreement.action == ACTIONS.MOVE:
        move_tree_services_pks = agreement.service.get_descendants(include_self=True).values_list('pk', flat=True)

    tree_services = {
        s.pk: s for s in Service.objects.filter(
            Q(pk__in=tree_services_pks) | Q(pk__in=move_tree_services_pks)
        )
    }

    oebs_products = get_oebs_products(tree_services)

    closure_links = Service._closure_model.objects.filter(parent_id__in=tree_services, depth=1)
    services_to_children = defaultdict(list)
    if agreement.action == ACTIONS.MOVE:
        closure_links = closure_links.exclude(child=agreement.service.pk)
        services_to_children[agreement.move_request.destination_id].append(agreement.service_id)
    for link in closure_links:
        services_to_children[link.parent_id].append(link.child_id)

    return build_oebs_subtree(root_service.pk, tree_services, services_to_children, oebs_products, agreement)[0]


def oebs_approvers_search(
        oebs_agreement: 'OEBSAgreement', role: Role,
        service_with_ancestors: Iterable['Service'],
        approvers_from_settings: list = None
):
    """
    Поиск апруверов:
        * ищем в текущем и вышестоящих сервисах сотрудников с необходимой ролью;
        * если там не находим, то если в параметрах указана настройка approvers_from_settings, берем оттуда
        * если в параметрах не указана настройка или указан пустой список, то бросаем ошибку.
    """

    for service in service_with_ancestors:
        members_with_roles = service.members.select_related('staff__login').filter(
            role=role,
        ).values_list('staff__login', flat=True)

        if members_with_roles:
            return list(members_with_roles)

    if approvers_from_settings:
        return approvers_from_settings

    notify_agreement_staff_not_found(agreement=oebs_agreement, role=role)


def get_approvers_abc_side():
    from plan.services.models import ServiceMember

    members = ServiceMember.objects.filter(
        service__slug='abc',
        role__code='responsible',
    ).select_related('staff')
    return [member.staff.login for member in members]


def _get_vs(service: 'Service') -> list:
    return service.get_ancestors(
        include_self=True
    ).prefetch_related('tags').filter(
        Q(is_base=True) | Q(tags__slug=settings.GRADIENT_VS)
    ).distinct().order_by('-level').values_list('slug', flat=True)


def _get_vs_approvers(service_slug: str) -> list:
    from plan.services.models import ServiceMember

    return [
        member.staff.login for member in
        ServiceMember.objects.filter(
            role__code=settings.OEBS_VS_ROLE_CODE,
            service__slug=service_slug,
        ).select_related('staff')
    ]


def get_move_approvers(oebs_agreement, abc_approvers):
    # https://st.yandex-team.ru/ABC-12672
    approvers = []
    if oebs_agreement.action == ACTIONS.MOVE:
        current_vs_ids = _get_vs(oebs_agreement.service)
        target_vs_ids = _get_vs(oebs_agreement.move_request.destination)
        if (current_vs_ids or target_vs_ids) and current_vs_ids != target_vs_ids:
            for service_slug in current_vs_ids:
                current_vs_approvers = _get_vs_approvers(service_slug=service_slug)
                if current_vs_approvers:
                    approvers.append(current_vs_approvers)
                    break

            for service_slug in target_vs_ids:
                target_vs_approvers = _get_vs_approvers(service_slug=service_slug)
                if target_vs_approvers:
                    approvers.append(target_vs_approvers)
                    break

            if not approvers:
                log.warning(f'Dont find VS approvers for {oebs_agreement.id}')
                approvers = [abc_approvers]
    return approvers


def get_oebs_approvers(oebs_agreement) -> Optional[list]:
    abc_approvers = get_approvers_abc_side()
    OEBS_MATCHING_FLAGS_AND_APPROVERS = {
        OEBS_HARDWARE_FLAG: settings.OEBS_HARDWARE_APPROVERS or abc_approvers,
        OEBS_HR_FLAG: settings.OEBS_HR_APPROVERS or abc_approvers,
        OEBS_PROCUREMENT_FLAG: settings.OEBS_PROCUREMENT_APPROVERS or abc_approvers,
        OEBS_REVENUE_FLAG: None,
    }
    approvers = []
    flags = set()

    if oebs_agreement.service.gradient_type() in [settings.GRADIENT_VS, settings.BUSINESS_UNIT_TAG]:
        approvers.append(abc_approvers)

    for flag in OEBS_FLAGS:
        # Если флаг меняется или действие не изменение флагов
        # и флаг стоит в True - нужно добавить
        # соответствующего согласующего
        from_attrs = oebs_agreement.attributes.get(flag)
        from_service = getattr(oebs_agreement.service, flag)
        if oebs_agreement.action == ACTIONS.CHANGE_FLAGS:
            # если действие изменение флагов и флаг не меняется
            # согласующий этот нам не нужен
            if from_attrs == from_service or from_attrs is None:
                continue
        if any((from_attrs, from_service)):
            flags.add(flag)

    if OEBS_GROUP_ONLY_FLAG in flags:
        # если выставляется этот флаг, согласующие берутся с других
        # флагов, которые стоят в True
        for flag in OEBS_FLAGS:
            from_service = getattr(oebs_agreement.service, flag)
            if from_service:
                flags.add(flag)

    if not flags:
        # сам сервис не oebs синхронизированный, нужно у потомков поискать
        descendants = (
            oebs_agreement.service
            .get_descendants(include_self=False)
            .filter(
                Q(use_for_hardware=True)|
                Q(use_for_hr=True)|
                Q(use_for_procurement=True)|
                Q(use_for_revenue=True)
            )
            .values_list('use_for_hardware', 'use_for_hr', 'use_for_procurement', 'use_for_revenue')
        )
        for use_for_hardware, use_for_hr, use_for_procurement, use_for_revenue in descendants:
            if use_for_hardware:
                flags.add(OEBS_HARDWARE_FLAG)
            if use_for_hr:
                flags.add(OEBS_HR_FLAG)
            if use_for_procurement:
                flags.add(OEBS_PROCUREMENT_FLAG)
            if use_for_revenue:
                flags.add(OEBS_REVENUE_FLAG)
            if len(flags) == len(OEBS_FLAGS):
                break

    if not flags:
        # такого не должно быть, произошла ошибка
        oebs_agreement.fail(
            ERRORS.ABC_ERROR, {'message': 'не удалось найти флаги для согласования'}
        )
        return approvers

    oebs_role = {
        role.code: role for role in Role.objects.filter(code__in=OEBS_MATCHING_FLAGS_AND_ROLES.values())
    }

    service_with_ancestors = oebs_agreement.service.get_ancestors(
        include_self=True
    ).order_by('-level')

    for flag in OEBS_FLAGS:
        if flag in flags:
            if flag == OEBS_GROUP_ONLY_FLAG:
                # у этого флага нет отдельных согласующих
                continue
            role_code = OEBS_MATCHING_FLAGS_AND_ROLES[flag]
            approvers_for_flag = oebs_approvers_search(
                oebs_agreement,
                oebs_role[role_code],
                service_with_ancestors,
                approvers_from_settings=OEBS_MATCHING_FLAGS_AND_APPROVERS[flag]
            )

            if approvers_for_flag is None:
                return None

            approvers.append(approvers_for_flag)

    move_outside_vs_approvers = get_move_approvers(
        oebs_agreement=oebs_agreement,
        abc_approvers=abc_approvers,
    )
    if move_outside_vs_approvers:
        approvers.extend(move_outside_vs_approvers)

    distinct_approvers = []
    for item in approvers:
        if item not in distinct_approvers:
            distinct_approvers.append(item)

    return distinct_approvers


def create_oebs_approve_issue(oebs_agreement):
    if oebs_agreement.issue:
        log.error(f'Issue {oebs_agreement.issue} for OEBSAgreement already exists - {oebs_agreement.id}')
        return oebs_agreement.issue

    context = {
        'requester': oebs_agreement.requester,
        'service': oebs_agreement.service,
        'agreement': oebs_agreement,
        'target_flags': oebs_agreement.get_target_flags(),
        'oebs_resource': oebs_agreement.service.oebs_resource,
        'agreement_tree_link': f'{settings.ABC_URL}/embed/oebs/tree/{oebs_agreement.id}/',
        'actions': ACTIONS,
    }
    approve_issue = create_issue(
        queue=settings.OEBS_APPROVE_QUEUE,
        summary=f'Согласование изменений в {oebs_agreement.service.name}',
        description=render_to_string('oebs_approve_ticket.txt', context),
        createdBy=oebs_agreement.requester.login,
        components=settings.OEBS_TICKET_COMPONENTS,
    )
    change_state(key=approve_issue.key, transition=settings.OEBS_TICKET_NEED_INFO)
    oebs_agreement.issue = approve_issue.key
    oebs_agreement.save(update_fields=['issue'])


def create_oebs_resource(oebs_agreement, leaf_oebs_id: str, parent_oebs_id: str):
    """
    Создаем oebs-ресурс
    """
    from plan.resources.models import Resource, ServiceResource, ResourceType   # тк импорты циклятся

    resource = Resource.objects.create(
        type=ResourceType.objects.get(code=settings.OEBS_PRODUCT_RESOURCE_TYPE_CODE),
        attributes={
            'parent_oebs_id': parent_oebs_id,
            'leaf_oebs_id': leaf_oebs_id,
        },
    )

    return ServiceResource.objects.create_granted(
        service=oebs_agreement.service,
        resource=resource,
        type_id=resource.type_id,
        attributes={
            'parent_oebs_id': parent_oebs_id,
            'leaf_oebs_id': leaf_oebs_id,
        },
    )


def oebs_request_failed(request):
    if request:
        oebs_agreement = request.oebsagreement_set.last()
        if oebs_agreement and oebs_agreement.state in STATES.UNSUCCESSFUL_STATES:
            fail_service_request(oebs_agreement)
            return True
    return False


def finalize_flag_changing(agreement):
    """
    Обновляем флаги Service и соответствующие им теги
    """
    from plan.services.models import Service, ServiceTag

    service = agreement.service
    to_add = []
    to_remove = []
    for flag in OEBS_FLAGS:
        new_value = agreement.attributes.get(flag)
        if new_value is False:
            to_remove.append(OEBS_FLAGS[flag])
        elif new_value is True:
            to_add.append(OEBS_FLAGS[flag])
    if to_add:
        service.tags.add(*ServiceTag.objects.filter(slug__in=to_add))
    if to_remove:
        service.tags.remove(*ServiceTag.objects.filter(slug__in=to_remove))

    flags = agreement.get_target_flags()
    Service.objects.filter(pk=service.id).update(**flags)


def get_assignee_for_renaming():
    from plan.duty.models import Shift

    shift = Shift.objects.filter(
        schedule_id=settings.OEBS_RENAMING_DUTY
    ).current_shifts().first()
    if shift and shift.staff:
        return shift.staff.login
    return settings.OEBS_RENAMING_DEFAULT


@register.filter
def get_item(dictionary, key):
    return dictionary.get(key)


def add_money_map_data(agreement):
    from plan.services.models import ServiceMember
    from plan.resources.models import ResourceType

    if (
        agreement.action == ACTIONS.MOVE
        and agreement.service.tags.filter(slug=settings.MONEY_MAP_TAG).exists()
    ):
        summonees = ServiceMember.objects.filter(
            service__in=agreement.service.get_ancestors(include_self=True),
            role__code=settings.MONEY_MAP_ROLE_CODE,
        ).select_related('staff')
        link = f'https://{settings.ABC_HOST}/services/{agreement.service.slug}/resources/?view=consuming&layout=table'
        for resource_type in ResourceType.objects.filter(code__in=settings.MONEY_MAP_RESOURCE_CODES):
            link += f'&supplier={resource_type.supplier_id}&type={resource_type.id}'
        create_comment(
            key=agreement.issue,
            comment_text=MONEY_MAP_COMMENT.format(link),
            summonees=[obj.staff.login for obj in summonees],
        )
