"""
Ping management command.
Prints '0;OK' if system appears to be working.
Prints '2;{error description} in case of error.
"""
from requests import get

from django.conf import settings
from django.core.management.base import BaseCommand


PING_MAX_LAG_IN_SECONDS = 5 * 60
PING_URL = 'https://127.0.0.1/ping'


class PingException(Exception):
    pass


class WrongStatus(PingException):
    """
    wrong http status code in response
    """
    def __str__(self):
        return 'Status code != 200, it\'s {0}!'.format(self.args[0])


def validate(response):
    """
    Validate ping handle response:
        - http status must be 200

    @type response: requests.Response
    @param response: ping handle http response
    @raise WrongStatus
    """
    if response.status_code != 200:
        raise WrongStatus(response.status_code)


class Command(BaseCommand):
    help = 'Ping service, validate response and return error in ' \
           'monrun-compliant format'

    def handle(self, *args, **options):
        try:
            headers = {'Host': settings.ABC_HOST}
            response = get(PING_URL, headers=headers, verify=False)
            validate(response)
        except PingException as exc:
            print('2;{0}'.format(exc))

        except Exception as exc:
            print('2;Unhandled exception:{0}'.format(exc))

        else:
            print('0;OK')
