import logging

from typing import Optional
from urllib.parse import urljoin, urlparse, parse_qs
from django.conf import settings

from plan.common.utils.http import Session

logger = logging.getLogger(__name__)


class PuncherClient:
    def __init__(self):
        self.host = settings.PUNCHER_URL

    def _make_response(self, url: str, params: dict) -> dict:
        with Session(oauth_token=settings.PUNCHER_ROBOT_TOKEN) as session:
            response = session.get(
                url=url,
                params=params,
            )
            response.raise_for_status()
        return response.json()

    def get_rules(self, service_id: str, cursor_id: Optional[str] = None) -> tuple[list, int, str]:
        params = {
            'service_id': service_id,
            'rules': 'exclude_rejected',
        }
        if cursor_id:
            params['_cursor_last_id'] = cursor_id

        url = urljoin(self.host, '/api/dynfw/rules')
        next_cursor_id = None
        response_data = self._make_response(url, params)
        result = response_data['rules']
        next_url = response_data['links'].get('next')
        if next_url:
            parsed_url = urlparse(next_url)
            next_cursor_id = parse_qs(parsed_url.query)['_cursor_last_id'][0]

        return result, response_data['count'], next_cursor_id

    def get_rules_count(self, service_id: int) -> int:
        url = urljoin(self.host, '/api/dynfw/rules')
        params = {
            'service_id': service_id,
            'rules': 'exclude_rejected',
        }
        response_data = self._make_response(url, params)
        return response_data['count']
