from django.contrib import admin

from plan.common.base_admin import ViewModelAdmin
from plan.resources import models
from plan.resources.tasks import grant_resource


@admin.register(models.ResourceTypeCategory)
class ResourceTypeCategoryAdmin(ViewModelAdmin):
    list_display = ('name', 'slug')


@admin.register(models.ResourceTagCategory)
class ResourceTagCategoryAdmin(ViewModelAdmin):
    list_display = ('name', 'slug', 'order', 'is_hidden')


@admin.register(models.ResourceTag)
class ResourceTagAdmin(ViewModelAdmin):
    list_display = ('name', 'slug', 'type', 'service')

    fieldsets = (
        (None, {
            'fields': (
                'name',
                'name_en',
                'slug',
                'type',
                'category',
                'service',
            )
        }),
        ('Описание', {
            'classes': ('collapse',),
            'fields': (
                ('description', 'description_en'),
            )
        }),
    )


@admin.register(models.ResourceType)
class ResourceTypeAdmin(ViewModelAdmin):
    list_display = ('name', 'supplier', 'import_plugin', 'category', 'is_enabled')
    list_filter = ('is_enabled', 'category')

    fieldsets = (
        (None, {
            'fields': (
                'supplier',
                'name',
                'code',
                'import_link',
                'import_plugin',
                'import_handler',
                'supplier_plugin',
                'form_id',
                'form_type',
                'form_handler',
                'form_back_handler',
                'usage_tag',
                'dependencies',
                'has_editable_tags',
                'has_tags',
                'has_supplier_tags',
                'has_multiple_consumers',
                'idempotent_request',
                'approve_policy',
                'supplier_roles',
                'supplier_scopes',
                'owner_roles',
                'owner_scopes',
                'consumer_roles',
                'consumer_scopes',
                'consumer_permissions',
                'is_enabled',
                'is_important',
                'is_immutable',
                'has_automated_grant',
                'category',
                'tags',
                'service_tags',
            )
        }),
        ('Описание', {
            'classes': ('collapse',),
            'fields': (
                ('description', 'description_en'),
            )
        }),
    )


class ServicesInline(admin.TabularInline):
    model = models.ServiceResource
    raw_id_fields = ('service', 'requester', 'obsolete')
    fields = (
        'id', 'service', 'requester',
        'obsolete', 'state', 'attributes',
    )


@admin.register(models.Resource)
class ResourceAdmin(ViewModelAdmin):
    list_display = ('name', 'type', 'external_id', 'parent', 'attributes',
                    'link')
    list_filter = ('type',)
    inlines = (ServicesInline,)


class ResourceTypeFilter(admin.SimpleListFilter):
    title = 'robots'
    parameter_name = 'resource__type'

    def lookups(self, request, model_admin):
        return (
            ('all', 'Все'),
            ('robots', 'Роботы'),
        )

    def queryset(self, request, queryset):
        qs = queryset
        if self.value() == 'robots':
            qs = queryset.filter(resource__type__code='staff-robot')
        return qs


@admin.register(models.ServiceResource)
class ServiceResourceAdmin(ViewModelAdmin):
    list_display = ('resource', 'service', 'state')
    list_filter = ('state', ResourceTypeFilter)
    search_fields = (
        'resource__name',
        'resource__external_id',
        'service__name',
        'service__slug',
    )

    actions = [
        'retry_grant',
    ]

    def retry_grant(self, request, queryset):
        for service_resource in queryset:
            grant_resource.delay(service_resource.id)

    retry_grant.short_description = 'Попробовать выдать ресурс (только из статуса Выдаётся)'
