import hashlib

from django.conf import settings
from rest_framework.status import HTTP_201_CREATED

from plan.api.exceptions import BadRequest, ABCAPIException
from plan.services.models import Service


class Created(ABCAPIException):
    status_code = HTTP_201_CREATED
    default_detail = 'created'
    default_message = {'ru': 'Создан', 'en': 'Created'}
    default_code = 'created'


def get_services(pk_or_slug, with_childs=True):
    try:
        service = Service.objects.get_by_pk_or_slug(pk_or_slug)
    except Service.DoesNotExist:
        return []

    if not with_childs:
        return [service]

    return service.get_descendants(include_self=True).active()


def make_signature(service, resource_type, user):
    data = [
        service.pk,
        resource_type.pk,
        user.username,
        settings.SECRET_KEY,
    ]
    s = '|'.join(map(str, data))
    return hashlib.sha256(s.encode()).hexdigest()


def validate_tag_affiliation(service, resource_type, tags=(), supplier_tags=()):
    '''Проверяем, можем ли мы проставить эти теги этой выдаче ресурса'''
    for tag in tags:
        if tag.service is not None and tag.service != service:
            raise BadRequest(message={
                'ru': 'Теги должны принадлежать тому же сервису, что и ресурс',
                'en': 'Tags must belong to the same service as the resource'
            })

    for tag in supplier_tags:
        if not (tag.service is None or tag.service == resource_type.supplier):
            raise BadRequest(message={
                'ru': 'Теги от поставщика ресурса должны принадлежать поставщику ресурса',
                'en': 'Supplier tags must belong to the supplier'
            })

    if resource_type.tags.exists():
        allowed_tags = resource_type.tags.all()

        for tag in supplier_tags:
            if tag not in allowed_tags:
                raise BadRequest(message={
                    'ru': 'Теги от поставщика ресурса должны соответствовать типу ресурса',
                    'en': "Supplier tags must match those of the resource's type"
                })
