import json
import logging

import requests

from django.conf import settings
from django.db import transaction
from django.utils.decorators import method_decorator
from django.utils.http import urlencode
from django.utils.datastructures import MultiValueDict
from django_replicated.decorators import use_master

from rest_framework import serializers, viewsets
from rest_framework.response import Response
from ylog.context import log_context

from plan.api import exceptions
from plan.common.utils.tvm import get_tvm_ticket
from plan.resources import permissions
from plan.resources.api.base import make_signature
from plan.resources.api.validators import ConstructorUrlRequestValidator
from plan.resources.models import ResourceType, ResourceTag, ServiceResource
from plan.resources.permissions import can_edit_resource, resource_is_active
from plan.resources.policies import BaseApprovePolicy
from plan.services.models import Service
from plan.swagger import SwaggerResources

log = logging.getLogger(__name__)


class ResourceGetFormSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=False
    )
    resource_type = serializers.PrimaryKeyRelatedField(
        queryset=ResourceType.objects.all(),
        required=False
    )
    service_resource = serializers.PrimaryKeyRelatedField(
        queryset=ServiceResource.objects.all(),
        required=False
    )
    tags = serializers.PrimaryKeyRelatedField(
        queryset=ResourceTag.objects.all(),
        many=True,
        required=False,
        default=list
    )
    supplier_tags = serializers.PrimaryKeyRelatedField(
        queryset=ResourceTag.objects.all(),
        many=True,
        required=False,
        default=list,
    )

    class Meta:
        validators = [
            ConstructorUrlRequestValidator()
        ]

    def validate(self, attrs):
        service_resource = attrs.get('service_resource')
        if service_resource:
            attrs['resource_type'] = service_resource.type

        return attrs


class ResourceRequestSerializer(ResourceGetFormSerializer):
    data = serializers.DictField()

    class Meta:
        validators = [
            ConstructorUrlRequestValidator()
        ]


def build_form_params(user, service=None, resource_type=None, service_resource=None,
                      tags=(), supplier_tags=(), use_iframe=True):
    """Конструирует для формы общие параметры
    """
    query_params = MultiValueDict()

    if service_resource is not None:
        query_params['service_resource'] = service_resource.id
        service = service_resource.service
        resource_type = service_resource.type

    signature = make_signature(
        service=service,
        resource_type=resource_type,
        user=user,
    )

    query_params.update({
        'service': service.id,
        'service_slug': service.slug,
        'service_name': service.name,
        'resource_type': resource_type.id,
        'tags': ','.join([str(tag.id) for tag in tags]),
        'supplier_tags': ','.join([str(tag.id) for tag in supplier_tags]),
        'signature': signature,
    })

    if use_iframe:
        query_params['iframe'] = '1'

    return query_params


def build_form_fields(resource_type, service_resource=None, data=None):
    """Конструирует для формы заполненные поля
    """
    context = {}

    if service_resource is not None:
        context.update(dict(
            service_resource.resource.attributes,
            service_resource=service_resource,
            resource_name=service_resource.resource.name
        ))

    if data is not None:
        context.update(data)

    return resource_type.process_form_back(**context) if context else {}


def build_constructor_url(resource_type, user, service_resource=None, data=None, **kwargs):
    """Конструирует полный урл до формы с общими параметрами и заполнеными полями
    """

    query_params = build_form_params(
        user=user,
        resource_type=resource_type,
        service_resource=service_resource,
        **kwargs,
    )
    query_params.update(build_form_fields(resource_type, service_resource, data))

    return resource_type.form_link + '?' + urlencode((k, vv) for k, v in query_params.lists() for vv in v)


@method_decorator(transaction.non_atomic_requests, name='dispatch')
@method_decorator(use_master, name='dispatch')
class ConstructorFormView(viewsets.ViewSet):
    """
    Запрос ресурса с заполнением формы
    """

    def _check_permissions(self, person, service, service_resource, resource_type):
        # если service_resource в запросе уже существует,
        # значит это фронт у нас просит отредактировать существующий,
        # а не запросить новый
        if service_resource:
            if not service_resource.service.is_exportable:
                raise exceptions.ServiceNotExportable()

            if not resource_is_active(service_resource):
                raise exceptions.ResourceTypeDisabled()

            resource_policy = BaseApprovePolicy.get_approve_policy_class(service_resource)(service_resource)
            if not can_edit_resource(person, resource_policy):
                raise exceptions.ResourceEditNoRoles()

            if not resource_type.form_back_handler:
                raise exceptions.ResourceTypeWithoutEdit()
        else:
            if not service.is_exportable:
                raise exceptions.ServiceNotExportable()

            if not resource_type.is_enabled:
                raise exceptions.ResourceTypeDisabled()

            if not permissions.can_request_resource(person, service, resource_type):
                raise exceptions.ResourceRequestNoRoles()

    def list(self, request):
        log.info('Request constructor form link')

        serializer = ResourceGetFormSerializer(data=request.GET)
        serializer.is_valid(raise_exception=True)
        data = serializer.validated_data

        service = data.get('service')
        resource_type = data.get('resource_type')
        service_resource = data.get('service_resource')
        self._check_permissions(request.person, service, service_resource, resource_type)

        url = build_constructor_url(user=request.user, **data) if resource_type.form_id else None
        return Response({'form_url': url})

    def create(self, request):
        """
        Пример заполения resource_type и data  для разных форм
        *Набор полей может со временем меняться*
        * Роботы в тестинге:
            ```
            resource_type: 92
            data: {'robot-login': < login >}
            ```
        * YP квоты в тестинге:
            ```
            resource_type: 164
            data: {'location': < loc >, 'segment': < loc >, 'cpu': < cpu >, 'memory': < mem >, 'hdd': < hdd >, 'ssd': < ssd >, 'ip4': < ip4 >}
            ```
        * TVM приложения:
            - В тестинге:
            ```
            resource_type: 54
            data: {"resource_name": "tvm-api-name"}
            ```
            - В продакшене:
            ```
            resource_type: 47
            data: {"resource_name": "tvm-api-name"}
            ```
        """

        serializer = ResourceRequestSerializer(data=request.data)
        serializer.is_valid(raise_exception=True)

        service = serializer.validated_data.get('service')
        resource_type = serializer.validated_data.get('resource_type')
        service_resource = serializer.validated_data.get('service_resource')
        self._check_permissions(request.person, service, service_resource, resource_type)

        data = serializer.validated_data.pop('data')
        query_params = build_form_params(
            request.user,
            use_iframe=False,
            **serializer.validated_data
        )

        built_fields = build_form_fields(resource_type, serializer.validated_data.get('service_resource'), data)
        filled_fields = {
            key: ('', str(value)) for key, value in
            built_fields.items()
        }

        data = {
            'source_request': json.dumps({
                'headers': {
                    'X_ABC_API': '1'
                },
                'query_params': query_params
            })
        }

        headers = {
            'Authorization': request.META.get('HTTP_AUTHORIZATION')
        }

        user_ticket = self.request.tvm_user_ticket
        if user_ticket:
            headers['X-Ya-Service-Ticket'] = get_tvm_ticket(destination=str(settings.CONSTRUCTOR_FORM_TVM_ID))
            headers['X-Ya-User-Ticket'] = user_ticket

        response = requests.post(
            resource_type.form_api_link,
            headers=headers,
            verify=settings.ABC_CA_ROOT_PATH,
            data=data,
            files=filled_fields
        )

        if response.status_code == 400:
            try:
                response_data = response.json()
            except Exception:
                log.exception(
                    'Forms badly answered with status 400 and content %s for fields %s',
                    response.status_code,
                    response.content[:10000],
                    json.dumps(filled_fields)
                )
                return Response(data={'errors': 'Forms answered with unparseable data'}, status=500)

            errors = {}
            for field_name, details in response_data.get('fields', {}).items():
                if details.get('errors'):
                    errors[field_name] = details.get('errors')
            if response_data.get('non_field_errors'):
                errors['non_field_errors'] = response_data['non_field_errors']

            with log_context(errors=errors):
                log.error(
                    'Forms answered with %d status and content %s for fields %s',
                    response.status_code,
                    json.dumps(errors),
                    json.dumps(filled_fields)
                )

            if response.status_code == 400:
                return Response(data={'errors': errors, 'created': False}, status=400)

        try:
            response.raise_for_status()
        except Exception:
            log.exception(
                'Forms answered with status %s and content %s for fields %s',
                response.status_code,
                response.content[:10000],
                json.dumps(filled_fields)
            )
            raise

        return Response(data=response.json(), status=response.status_code)


@method_decorator(transaction.non_atomic_requests, name='dispatch')
@method_decorator(use_master, name='dispatch')
class V4ConstructorFormView(ConstructorFormView):
    """
    Запрос ресурса с заполнением формы
    """
    default_swagger_schema = SwaggerResources
