import django_filters
from rest_framework import viewsets

from plan.api.filters import PlanFilterSet
from plan.api.mixins import OrderingMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.api.base import ABCCursorPagination
from plan.api.serializers import (
    CompactResourceTypeSerializer,
    ModelSerializer,
)
from plan.resources.models import ServiceResourceCounter
from plan.swagger import SwaggerFrontend


class ServiceResourceCounterFilter(PlanFilterSet):
    service_id = django_filters.CharFilter()
    resource_type_id = django_filters.CharFilter()

    class Meta:
        model = ServiceResourceCounter
        fields = ('service_id', 'resource_type_id', )


class ServiceResourceCounterSerializer(ModelSerializer):
    resource_type = CompactResourceTypeSerializer()

    class Meta:
        model = ServiceResourceCounter
        fields = ('id', 'service_id', 'resource_type', 'count', )


class ServiceResourceCounterView(OrderingMixin, TvmAccessMixin, viewsets.ReadOnlyModelViewSet):
    """
    Счетчики количества ServiceResource в ALIVE_STATES
    """
    default_swagger_schema = SwaggerFrontend

    _permissions_to_proceed = 'can_view'
    serializer_class = ServiceResourceCounterSerializer
    permission_classes = [TvmAuthenticated]
    filter_class = ServiceResourceCounterFilter
    pagination_class = ABCCursorPagination
    queryset = ServiceResourceCounter.objects.select_related(
        'resource_type', 'resource_type__supplier',
    )
