import logging

from django.conf import settings
from django.utils import timezone

from plan.api.exceptions import PermissionDenied
from rest_framework import mixins, serializers, viewsets


from plan.resources.handlers.financial_resources.direct.forward import _make_unique_obj_id
from plan.resources.models import Resource, ResourceType, ServiceResource
from plan.resources.permissions import can_request_resource_api
from plan.resources.tasks import grant_resource
from plan.roles.models import Role
from plan.services.models import Service, ServiceMember
from plan.staff.models import Staff
from plan.swagger import SwaggerResources

log = logging.getLogger(__name__)


FINANCIAL_RESOURCE_ATTRS_TYPE = {
    settings.BALANCE_RESOURCE_TYPE_CODE: {'client_id': str},
    settings.DIRECT_RESOURCE_TYPE_CODE: {'client_id': str},
}


class FinancialResourceRequestSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.active().exportable(),
        required=True,
    )
    resource_type = serializers.PrimaryKeyRelatedField(
        queryset=ResourceType.objects.all(),
        required=True,
    )
    obj_id = serializers.IntegerField(required=True)
    attributes = serializers.JSONField(required=False)


class FinancialResourceRequestView(mixins.CreateModelMixin, viewsets.GenericViewSet):
    serializer_class = FinancialResourceRequestSerializer

    def create_service_resource(self, resource: Resource, service_id: int):
        return ServiceResource.objects.create(
            resource=resource,
            service_id=service_id,
            state=ServiceResource.GRANTING,
        )

    def update_or_create_service_resource(self, resource: Resource, service: Service, requester: Staff) -> ServiceResource:
        """Create or restore ServiceResources"""

        service_resources = ServiceResource.objects.filter(
            resource=resource,
            service=service,
        ).exclude(state=ServiceResource.OBSOLETE)
        if service_resources.exists():
            granted = service_resources.filter(
                state__in=(ServiceResource.GRANTED, ServiceResource.GRANTING)
            ).order_by('state')
            if granted.exists():
                service_resource = granted.first()
            else:
                service_resource = service_resources.order_by('-created_at').first()
                service_resource.state = ServiceResource.GRANTING
                service_resource.save(update_fields=['state'])
        else:
            service_resource = ServiceResource.objects.create(
                resource=resource,
                service=service,
                type=resource.type,
                state=ServiceResource.GRANTING,
                requester=requester,
            )

        return service_resource

    def edit_resource(self, resource: Resource, attributes: dict):
        now = timezone.now()
        new_resource = Resource.objects.create(
            external_id=resource.external_id,
            name=resource.name,
            type_id=resource.type_id,
            attributes=attributes,
            obsolete=resource,
        )
        old_service_resources = ServiceResource.objects.filter(resource=resource)
        old_active_service_resources = old_service_resources.filter(
            state__in=(ServiceResource.GRANTING, ServiceResource.GRANTED)
        )
        for old_service_resource in old_active_service_resources:
            ServiceResource.objects.create(
                service_id=old_service_resource.service_id,
                resource=new_resource,
                state=ServiceResource.GRANTING,
                obsolete=old_service_resource,
                type_id=new_resource.type_id,
            )

        old_active_service_resources.update(state=ServiceResource.OBSOLETE, modified_at=now)
        old_service_resources.exclude(state=ServiceResource.OBSOLETE).update(
            state=ServiceResource.DEPRIVED,
            modified_at=now,
        )

        return new_resource

    def _make_obj_id(self, validated_data, attributes, resource_type):
        obj_id = str(validated_data['obj_id'])
        if resource_type.code == settings.DIRECT_RESOURCE_TYPE_CODE:
            obj_id = _make_unique_obj_id(obj_id, attributes.get('main_manager'))

        return obj_id

    def perform_create(self, serializer: FinancialResourceRequestSerializer):
        resource_type = serializer.validated_data['resource_type']

        attributes = serializer.validated_data.get('attributes', {})
        service = serializer.validated_data['service']
        obj_id = self._make_obj_id(
            validated_data=serializer.validated_data,
            attributes=attributes,
            resource_type=resource_type
        )

        requester = self.request.person
        log.info(f'Request to create a service_resource by {requester} with parameters: {self.request.data}')
        if not (
            requester.user.has_perm('internal_roles.create_financial_resources')
            or ServiceMember.objects.filter(
                service=service,
                staff=requester.staff,
                role__code=Role.SERVICE_FINANCIAL_RESOURCES_REQUESTER,
            ).exists()
            or not can_request_resource_api(self.request.person, service, resource_type)
        ):
            raise PermissionDenied()

        if resource_type.code in FINANCIAL_RESOURCE_ATTRS_TYPE:
            for key, value in FINANCIAL_RESOURCE_ATTRS_TYPE[resource_type.code].items():
                if key in attributes:
                    attributes[key] = value(attributes[key])

        resource = Resource.objects.filter(type=resource_type, external_id=obj_id).order_by('-created_at').first()
        if resource is not None:
            if attributes and resource.attributes != attributes:
                resource = self.edit_resource(resource, attributes)

        else:
            resource = Resource.objects.create(type=resource_type, external_id=obj_id, attributes=attributes, name=obj_id)

        service_resource = self.update_or_create_service_resource(resource=resource, service=service, requester=requester.staff)
        log.info(f'Create {service_resource} with param: obj_id {obj_id}, attributes {attributes}, service {service}')
        grant_resource.apply_async(args=[service_resource.id])


class V4FinancialResourceRequestView(FinancialResourceRequestView):
    """
    Запрос ресурсов только для типов: direct_client, metrika_counter_id, balance_client
    ID типов:
    ```
    Тип     | Testing | Production
    Баланс  | 341     | 142
    Директ  | 306     | 138
    Метрика | 273     | 105
    ```
    """

    default_swagger_schema = SwaggerResources
