from requests import codes as http_codes

from django import forms

from rest_framework import generics, response

import django_filters

from plan.api import base
from plan.resources.models import Resource
from plan.roles.models import Role, RoleScope
from plan.services.models import ServiceMember, Service
from plan.staff.models import Staff
from plan.api.filters import PlanFilterSet, CustomModelChoiceFilter
from plan.api.serializers import BasePlanModelSerializer


class RoleSerializer(base.ModelSerializer):
    class Meta:
        model = Role
        fields = ['id', 'name', 'name_en', 'scope']


class ServiceMemberSerializer(BasePlanModelSerializer):
    service = base.CompactServiceSerializer()
    staff = base.CompactStaffSerializer()
    role = RoleSerializer()

    class Meta:
        model = ServiceMember
        fields = ['staff', 'service', 'role']


class ResourceWidget(forms.Widget):
    fields = ['resource_id', 'resource_type', 'resource_source']

    def value_from_datadict(self, data, files, name):
        value = [data.get(field, None) for field in self.fields]
        return value if all(value) else None


class ResourceField(forms.Field):
    """В текущей версии django (1.5) orm по разному понимает
    .filter(a).filter(b) и .filter(a, b). В первом случае orm генерирует не
    оптимальный запрос при фильтрации по many2many. Именно на таком создании
    запросов базируется rest_framework, поэтому мы не могли сделать один большой
    запрос в PermissionFilter, пришлось сначала выбирать ресурс, а потом
    фильтровать уже по первичному ключу. Поэтому пришлось создать этот Field
    """
    widget = ResourceWidget
    default_error_messages = {
        'required': 'resource_id, resource_type and resource_supplier fields '
                    'is required.',
    }

    def to_python(self, value):
        if value is None:
            return None

        resource_id, resource_type, resource_supplier = value
        try:
            return (
                Resource.objects.get(
                    external_id=resource_id,
                    type__name=resource_type,
                    type__supplier__slug=resource_supplier)
            )
        except Resource.DoesNotExist:
            raise forms.ValidationError('Resource does not exist')


class ResourceFilter(django_filters.Filter):
    field_class = ResourceField


class PermissionFilter(PlanFilterSet):
    resource = ResourceFilter(field_name='service__serviceresource__resource', required=True)
    user = CustomModelChoiceFilter(
        field_name='staff',
        to_field_name='login',
        queryset=Staff.objects.all(),
    )
    service = CustomModelChoiceFilter(
        field_name='service',
        to_field_name='slug',
        queryset=Service.objects.all(),
    )
    role = CustomModelChoiceFilter(
        field_name='role',
        queryset=Role.objects.all(),
    )
    role_scope = CustomModelChoiceFilter(
        field_name='role__scope',
        queryset=RoleScope.objects.all(),
    )

    @property
    def form(self):
        """APIView игнорирует наличие ошибок в форме фильтров, поэтому для
        валидации приходится бросать исключение руками
        """
        form = super(PermissionFilter, self).form
        if not form.is_valid():
            raise forms.ValidationError('validation error', params=form.errors)
        return form

    class Meta:
        model = ServiceMember
        fields = ['resource', 'user', 'service', 'role', 'role_scope']


class PermissionView(generics.ListAPIView):
    filter_class = PermissionFilter
    serializer_class = ServiceMemberSerializer
    queryset = (
        ServiceMember.objects
        .select_related(
            'role',
            'staff',
            'service'
        )
        .order_by('pk')
    )

    def handle_exception(self, exc):
        if isinstance(exc, forms.ValidationError):
            return response.Response(
                data={'errors': exc.params},
                status=http_codes.bad_request
            )

        return super(PermissionView, self).handle_exception(exc)


class AssertFilter(PlanFilterSet):
    resource = ResourceFilter(field_name='service__serviceresource__resource', required=True)
    user = django_filters.CharFilter(field_name='staff__login')
    service = django_filters.CharFilter(field_name='service__slug')
    role = django_filters.NumberFilter(field_name='role_id')
    role_scope = django_filters.NumberFilter(field_name='role__scope')

    class Meta:
        model = ServiceMember
        fields = ['resource', 'user', 'service', 'role', 'role_scope']


class AssertView(generics.GenericAPIView):
    filter_class = AssertFilter
    model = ServiceMember

    def get(self, request, *args, **kwargs):
        queryset = ServiceMember.objects.order_by('pk')
        objects_count = self.filter_queryset(queryset).count()
        return response.Response({'assertion': bool(objects_count)})
