import json
import logging
from urllib.parse import unquote

from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django.db.utils import IntegrityError
from django.utils.decorators import method_decorator
from django_replicated.decorators import use_master
from rest_framework import viewsets, serializers
from rest_framework.response import Response
from django.utils.translation import ugettext_lazy as _

from plan.api.exceptions import BadRequest, Conflict, PermissionDenied, ValidationError
from plan.common.person import Person
from plan.resources.api.base import make_signature, validate_tag_affiliation
from plan.resources.forms import RequestForm
from plan.resources.models import Resource, ServiceResource
from plan.resources.permissions import can_request_resource_api
from plan.resources.processing import ProcessorError
from plan.resources.suppliers.base import SupplierPlugin

log = logging.getLogger(__name__)


@method_decorator(use_master, name='dispatch')
class RequestResourceView(viewsets.GenericViewSet):
    serializer_class = serializers.Serializer

    def perform_authentication(self, request):
        pass

    def list(self, request, *args, **kwargs):
        """ Требование конструктора форм """
        return Response('')

    def get_request_data(self):
        result = {}
        for k, v in self.request.POST.items():
            if 'field' in k:
                data = json.loads(v)
                if isinstance(data, dict) and data.get('value') == 'None':
                    data['value'] = None

                result[k] = data

        return result

    def check_signature(self, data):
        signature = make_signature(
            service=data['service'],
            resource_type=data['resource_type'],
            user=data['user'],
        )
        if signature != self.request.META.get('HTTP_X_ABC_SIGNATURE'):
            raise ValidationError(detail='Signature mismatch')

    @staticmethod
    def _process_tags_string(tags):
        return unquote(tags).split(',') if tags else []

    def create(self, request, *args, **kwargs) -> Response:
        request_id = request.META.get('HTTP_X_FORM_ANSWER_ID')
        log.info('Resource request %s', request_id)

        tags = request.META.get('HTTP_X_ABC_RESOURCE_TAGS')

        supplier_tags = request.META.get('HTTP_X_ABC_RESOURCE_SUPPLIER_TAGS')
        form_data = {
            'user': request.META.get('HTTP_X_ABC_USER'),
            'service': request.META.get('HTTP_X_ABC_SERVICE'),
            'resource_type': request.META.get('HTTP_X_ABC_RESOURCE_TYPE'),
            'tags': self._process_tags_string(tags),
            'supplier_tags': self._process_tags_string(supplier_tags),
            'service_resource': request.META.get('HTTP_X_ABC_SERVICE_RESOURCE'),
        }

        # странный баг форм
        if form_data['service_resource'] == 'None':
            form_data['service_resource'] = None

        form = RequestForm(form_data)
        if not form.is_valid():
            raise ValidationError(extra=form.errors)

        resource_type = form.cleaned_data['resource_type']

        self.check_signature(form.cleaned_data)

        try:
            data = self.get_request_data()
        except ValueError as e:
            raise ValidationError(detail='Bad POST format: %s' % e)

        try:
            request_params = resource_type.process_form(data, form.cleaned_data)
        except ProcessorError as pe:
            error_message = getattr(pe.exception, 'message', 'Unknown error')
            raise BadRequest(message='Error during form processing', detail=error_message) from pe.exception

        if 'external_id' not in request_params:
            request_params['external_id'] = None

        if form.cleaned_data.get('service_resource'):
            return self.edit_resource(form.cleaned_data, request_params, request_id)
        else:
            return self.create_resource(form.cleaned_data, request_params, request_id)

    def edit_resource(self, form_data, request_params, request_id) -> Response:
        # Редактирование существующего.
        # Здесь права не проверяем, потому что проверили при подписывании ссылки на редактирование
        resource = form_data['service_resource'].resource

        if not resource.type.form_back_handler:
            message = {
                'ru': 'Извините, поставщик ресурсов этого типа не поддерживает их редактирование.',
                'en': 'Sorry, the supplier of resources of this type does not support editing them.'
            }
            raise BadRequest(
                code='resource_edit_disabled',
                detail='Resource type does not support editing.',
                message=message
            )

        if resource.type.is_immutable:
            self.replace_resource(form_data, request_params, request_id, old_resource=resource)
            return Response({'created': True})

        log.info('Edit resource %s by %s', resource, form_data['user'])

        resource.name = request_params.get('name', resource.name)
        resource.link = request_params.get('link', resource.link)
        resource.attributes = request_params.get('attributes', resource.attributes)

        if resource.type.supplier_plugin:
            plugin = SupplierPlugin.get_plugin_class(resource.type.supplier_plugin)()
            try:
                plugin.edit(form_data['service_resource'], form_data['user'])
            except Exception as ex:
                raise Conflict(detail=getattr(ex, 'message', str(ex)))

        resource.save()

        return Response({'edited': True})

    def _check_has_multiple_consumers_restrictions(self, resource_type, service, resource):
        if resource_type.code == settings.TVM_RESOURCE_TYPE_CODE:
            # проверим, что есть только один активный ресурс
            existing_resources = ServiceResource.objects.filter(resource=resource).alive()
            if existing_resources.count() > 1:
                raise BadRequest(message={
                    'ru': 'Уже есть ресурс с таким tvmid, находящийся в стадии утверждения',
                    'en': 'Approving resource with same tvmid already exists',
                })

            existing_resource = existing_resources.first()
            if not existing_resource:
                raise BadRequest(message={
                    'ru': 'Ресурса с таким tvmid не существует',
                    'en': 'There is no resource with this tvmid',
                })
            if existing_resource.service_id == service.id:
                raise BadRequest(message={
                    'ru': 'Ресурс с таким tvmid уже связан с данным сервисом',
                    'en': 'Resource with same tvmid is already associated with this service',
                })
            if existing_resource.state != ServiceResource.GRANTED:
                raise BadRequest(message={
                    'ru': 'Существующий ресурс должен быть утвержден для переноса',
                    'en': 'Existing resource must become approved before moving',
                })

        else:
            raise Conflict(message={
                'ru': 'Многократное использование ресурсов запрещено',
                'en': 'Supplier forbid multiple consumers',
            })

    @transaction.atomic()
    def create_resource(self, form_data, request_params, request_id) -> Response:
        # Новый запрос
        resource_type = form_data['resource_type']
        service = form_data['service']

        if not can_request_resource_api(Person(form_data['user']), service, resource_type):
            raise ValidationError(_('Вы не можете запросить этот ресурс'))

        resource = (
            Resource.objects
            .filter(type=resource_type, external_id=request_params['external_id'])
            .order_by('-created_at')  # будем надеяться, что это актуальный ресурс
        ).first()

        if request_params['external_id'] is not None and resource is not None:
            # Добавляем к ресурсу новый сервис
            if not resource_type.has_multiple_consumers:
                self._check_has_multiple_consumers_restrictions(
                    resource_type=resource_type,
                    service=service,
                    resource=resource,
                )

            if 'attributes' in request_params and request_params['attributes'] != resource.attributes:
                log.info('Updating attributes for resource %s', resource.external_id)
                if resource_type.is_immutable:
                    replace_params = {
                        'external_id': resource.external_id,
                        'name': resource.name,
                        'link': resource.link,
                        'attributes': request_params['attributes']
                    }
                    resource = self.replace_resource(form_data, replace_params, request_id, resource)
                else:
                    resource.attributes = request_params['attributes']
                    resource.save(update_fields=['attributes'])

            created = False

        else:
            # Создаем новый ресурс
            log.info('Create resource %s by %s', request_params['external_id'], form_data['user'])

            try:
                resource = Resource.objects.create(
                    type=resource_type,
                    external_id=request_params['external_id'],
                    name=request_params.get('name', ''),
                    link=request_params.get('link', ''),
                    attributes=request_params.get('attributes', {}),
                    answer_id=request_id,
                )
            except IntegrityError:
                log.warning(
                    'Constructor form is retrying resource {type_id: %s, external_id: %s, answer_id: %s}',
                    resource_type.id,
                    request_params['external_id'],
                    request_id,
                )
                raise Conflict(message={
                    'ru': 'Попытка создать уже существующий ресурс с external_id: %s' % request_params['external_id'],
                    'en': 'Attempt to create existing resource with external_id: %s' % request_params['external_id'],
                })
            created = True

        alike_qs = (
            ServiceResource.objects
            .filter(resource=resource, service=service)
            .filter(Q(state__in=ServiceResource.ALMOST_ACTIVE_STATES) | Q(request_id=request_id))
        )

        if not alike_qs.exists():
            log.info('Create service resource %s@%s by %s',
                     request_params['external_id'], form_data['service'], form_data['user'])

            try:
                service_resource = ServiceResource.objects.create(
                    resource=resource,
                    state=ServiceResource.REQUESTED,
                    service=service,
                    request_id=request_id,
                    requester=form_data['user'],
                    type_id=resource.type_id,
                )
            except IntegrityError:
                raise Conflict(message={
                    'ru': 'Ресурс уже привязан к этому сервису',
                    'en': 'Resource is already added to this service',
                })

            if resource_type.code == settings.TVM_RESOURCE_TYPE_CODE:
                # нужно найти прошлый активный ресурс и связать его с текущим
                old_serviceresource = resource.serviceresource_set.alive().exclude(pk=service_resource.pk).first()
                if old_serviceresource:
                    service_resource.obsolete = old_serviceresource
                    service_resource.save(update_fields=('obsolete', ))

            tags = form_data['tags']
            supplier_tags = form_data['supplier_tags']

            validate_tag_affiliation(service, resource_type, tags, supplier_tags)

            for tag in tags:
                service_resource.tags.add(tag.id)

            for tag in supplier_tags:
                service_resource.supplier_tags.add(tag.id)

            try:
                service_resource.approve(Person(form_data['user']), request=self.request)
            except (BadRequest, PermissionDenied):
                pass

        return Response({'created': created})

    @transaction.atomic
    def replace_resource(self, form_data, request_params, request_id, old_resource) -> Resource:
        if ServiceResource.objects.filter(request_id=request_id).exists():
            return old_resource

        # Замена ресурса
        log.info('Create new resource instead of %s by %s', old_resource, form_data['user'])

        # Создаем новый ресурс
        log.info('Create resource %s by %s', request_params['external_id'], form_data['user'])

        try:
            resource = Resource.objects.create(
                type=old_resource.type,
                external_id=request_params['external_id'],
                name=request_params.get('name', ''),
                link=request_params.get('link', ''),
                attributes=request_params.get('attributes', {}),
                obsolete=old_resource,
                answer_id=request_id,
            )
        except IntegrityError:
            raise Conflict(message={
                'ru': 'Ресурс с таким идентификатором уже существует',
                'en': 'Resource with this id already exists'
            })

        for old_serviceresource in old_resource.serviceresource_set.alive():
            log.info('Create service resource instead of %s by %s', old_serviceresource, form_data['user'])

            service_resource = ServiceResource.objects.create(
                resource=resource,
                state='requested',
                service=old_serviceresource.service,
                request_id=request_id,
                requester=form_data['user'],
                obsolete=old_serviceresource,
                type_id=resource.type_id,
            )

            for tag in old_serviceresource.tags.all():
                service_resource.tags.add(tag)

            for tag in old_serviceresource.supplier_tags.all():
                service_resource.supplier_tags.add(tag)

            try:
                service_resource.approve(Person(form_data['user']), request=self.request)
            except (BadRequest, PermissionDenied):
                pass

        return resource
