import django_filters

from rest_framework import viewsets

from plan.api import base
from plan.api.fields import MappingField
from plan.api.filters import PlanFilterSet
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.resources.models import ResourceTagCategory
from plan.swagger import SwaggerResources


class TagCategorySerializer(base.ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = ResourceTagCategory
        fields = [
            'id',
            'name',
            'slug',
        ]


class CategoryWithTagsFilter(django_filters.BooleanFilter):
    def filter(self, qs, value):
        if value:
            qs = qs.filter(resourcetag__isnull=False).distinct()

        return qs


class TagCategoryFilter(PlanFilterSet):
    with_tags = CategoryWithTagsFilter()

    class Meta:
        model = ResourceTagCategory
        fields = ('name', 'name_en', 'slug', 'with_tags')


class ResourceTagCategoryView(base.OrderingMixin, viewsets.ModelViewSet):

    serializer_class = TagCategorySerializer
    filter_class = TagCategoryFilter
    queryset = ResourceTagCategory.objects.all()
    http_method_names = ['get']


class V4ResourceTagCategoryView(DefaultFieldsMixin, ResourceTagCategoryView, TvmAccessMixin):
    """
    Категории тегов ресурсов
    """
    default_swagger_schema = SwaggerResources

    permission_classes = [TvmAuthenticated]
    default_fields = ['id', 'slug']
