import six

import django_filters
from django.db.models import Q
from django.forms import Form as DjangoForm
from rest_framework import serializers, viewsets, status
from rest_framework.metadata import BaseMetadata
from rest_framework.response import Response

from plan.api import base
from plan.api.exceptions import PermissionDenied, ValidationError
from plan.api.fields import MappingField, IntegerInterfacedSerializerField
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.api.serializers import BasePlanModelSerializer
from plan.resources.api.resource_tag_categories import TagCategorySerializer
from plan.resources.models import ResourceTag, ResourceTagCategory, ResourceType
from plan.services.models import Service
from plan.swagger import SwaggerResources


class TagSerializer(base.ModelSerializer):
    category = TagCategorySerializer()
    service = base.CompactServiceSerializer()
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    description = MappingField({'ru': 'description', 'en': 'description_en'})
    fields_mapping_ = {
        'name': ('name', 'name_en'),
        'description': ('description', 'description_en'),
    }

    class Meta:
        model = ResourceTag
        fields = (
            'id', 'name', 'slug', 'type', 'category', 'service', 'description'
        )


class CreateTagSerializer(BasePlanModelSerializer):
    category = serializers.PrimaryKeyRelatedField(
        queryset=ResourceTagCategory.objects.all(),
        required=False,
    )
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=True,
    )
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    description = MappingField({'ru': 'description', 'en': 'description_en'}, required=False)
    fields_mapping_ = {
        'name': ('name', 'name_en'),
        'description': ('description', 'description_en'),
    }

    class Meta:
        model = ResourceTag
        fields = ('slug',  'description', 'name', 'type', 'category', 'service')


class UpdateTagSerializer(BasePlanModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    description = MappingField({'ru': 'description', 'en': 'description_en'}, required=False)

    category = IntegerInterfacedSerializerField(
        queryset=ResourceTagCategory.objects.all(),
        serializer=TagCategorySerializer,
        allow_null=True,
    )

    class Meta:
        model = ResourceTag
        fields = ('slug', 'name', 'description', 'type', 'category')


class ResourceTagFilterForm(DjangoForm):
    def clean_category__is_hidden(self):
        if self.cleaned_data['category__is_hidden'] is None:
            self.cleaned_data['category__is_hidden'] = False
        return self.cleaned_data['category__is_hidden']


class ResourceTypeFilter(CustomModelMultipleChoiceFilter):

    def filter(self, qs, value):
        if not value or self.is_noop(qs, value):
            return qs
        prepared_value = [obj.pk for obj in value]
        filter_data = {'type_tags__id__in': prepared_value}
        filter_suplier_data = {'service__supplier__in': prepared_value}

        if qs.filter(**filter_data).exists():
            # Теги, привязанные к этому типу ресурса
            qs = qs.filter(**filter_data)

        else:
            # Теги, привязанные к поставщику этого типа или глобальные теги
            qs = qs.filter(Q(**filter_suplier_data) | Q(service=None))

        return qs.distinct() if self.distinct else qs


class TagCategoryIsHiddenFilter(django_filters.rest_framework.BooleanFilter):

    def filter(self, qs, value):
        if isinstance(value, django_filters.fields.Lookup):
            lookup = six.text_type(value.lookup_type)
            value = value.value
        else:
            lookup = self.lookup_expr
        if value in django_filters.constants.EMPTY_VALUES:
            return qs
        if self.distinct:
            qs = qs.distinct()

        if any([param.startswith(self.field_name) for param in self.parent.request.query_params.keys()]):
            query = Q(**{f'{self.field_name}__{lookup}': value})
        else:
            query = Q(**{f'{self.field_name}__{lookup}': value}) | Q(category__isnull=True)

        return self.get_method(qs)(query)


class TagFilter(PlanFilterSet):
    service = CustomModelMultipleChoiceFilter(
        field_name='service',
        queryset=Service.objects.all(),
    )
    category = CustomModelMultipleChoiceFilter(
        field_name='category',
        queryset=ResourceTagCategory.objects.all(),
    )
    resource_type = ResourceTypeFilter(
        queryset=ResourceType.objects.all(),
        lookup_expr='in',
    )
    category__is_hidden = TagCategoryIsHiddenFilter()

    class Meta:
        model = ResourceTag
        form = ResourceTagFilterForm
        fields = {
            'name': ['exact', 'contains'],
            'name_en': ['exact', 'contains'],
            'slug': ['exact', 'contains', 'in'],
            'type': ['exact'],
            'service': ['exact', 'isnull'],
            'category__is_hidden': ['exact'],
            'resource_type': ['exact'],
        }


class PermissionsMetadata(BaseMetadata):
    def determine_metadata(self, request, view):
        service_id = request.data.get('service')
        if service_id:
            service = Service.objects.get(id=service_id)
            permissions = []

            if request.person.is_member_of_team(service, with_ancestors=False):
                permissions.append('can_create_tags')
                permissions.append('can_edit_tags')

            return {'permissions': permissions}
        else:
            return {'permissions': []}


class ResourceTagView(base.OrderingMixin, viewsets.ModelViewSet):

    serializer_class = TagSerializer
    filter_class = TagFilter
    queryset = ResourceTag.objects.all().select_related('category', 'service')
    ordering = ('category__order', 'pk')
    http_method_names = ['get', 'post', 'put', 'patch', 'delete', 'options']
    metadata_class = PermissionsMetadata

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateTagSerializer

        if self.action in ('update', 'partial_update'):
            return UpdateTagSerializer

        return TagSerializer

    def check_permissions(self, request, service=None):
        # редактировать теги ресурсов могут только те,
        # кому можно редактировать ресурсы сервиса-хозяина тега
        if self.action == 'create':
            if service is None:
                serializer = self.get_serializer(data=request.data)
                if serializer.is_valid():
                    service = serializer.validated_data['service']
                else:
                    return

            message = {
                'ru': 'Вам нельзя добавлять теги ресурсов к этому сервису',
                'en': 'You are not allowed to add resource tags to this service'
            }

        if self.action in ('update', 'partial_update', 'destroy'):
            service = self.get_object().service
            message = {
                'ru': 'Вам нельзя редактировать этот тег ресурсов',
                'en': "You can't edit this resource tag"
            }

        restricted = self.action in ('create', 'update', 'partial_update', 'destroy')
        if restricted and not request.person.is_member_of_team(service, with_ancestors=False):
            raise PermissionDenied(message=message)

    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        if not serializer.is_valid():
            raise ValidationError(extra=serializer.errors)

        self.check_permissions(request, service=serializer.validated_data['service'])

        self.perform_create(serializer)
        headers = self.get_success_headers(serializer.data)
        return Response(serializer.data, status=status.HTTP_201_CREATED, headers=headers)

    def update(self, request, *args, **kwargs):
        self.check_permissions(request)
        return super(ResourceTagView, self).update(request, *args, **kwargs)

    def partial_update(self, request, *args, **kwargs):
        self.check_permissions(request)
        return super(ResourceTagView, self).partial_update(request, *args, **kwargs)

    def destroy(self, request, *args, **kwargs):
        self.check_permissions(request)
        return super(ResourceTagView, self).destroy(request, *args, **kwargs)


class V4ResourceTagView(DefaultFieldsMixin, ResourceTagView, TvmAccessMixin):
    """
    Теги ресурсов
    """
    default_swagger_schema = SwaggerResources

    permission_classes = [TvmAuthenticated]
    default_fields = [
        'id',
        'slug',
        'type',

        'category.id',
        'category.slug',
    ]
