from django.utils.translation import ugettext_lazy as _

import django_filters

from rest_framework import viewsets, serializers
from rest_framework.decorators import action
from rest_framework.generics import get_object_or_404
from rest_framework.response import Response

from plan.api import base
from plan.api.base import ABCCursorPagination
from plan.api.exceptions import NotFound
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.fields import MappingField, IntegerInterfacedSerializerField
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.serializers import CompactResourceTypeSerializer
from plan.api.permissions import TvmAuthenticated
from plan.resources.api.base import get_services
from plan.resources.api.resource_tags import TagSerializer
from plan.resources.api.resource_type_categories import TypeCategorySerializer
from plan.resources.models import ResourceTag, ResourceType, ResourceTypeCategory
from plan.roles.api.roles import RoleSerializer
from plan.roles.models import RoleScope, merge_allowed_roles
from plan.services.models import Service
from plan.swagger import SwaggerResources


class TypeSerializer(base.ModelSerializer):
    supplier = base.CompactServiceSerializer()
    category = TypeCategorySerializer()
    name = MappingField({'ru': 'name', 'en': 'name'})
    description = MappingField({'ru': 'description', 'en': 'description_en'})
    tags = IntegerInterfacedSerializerField(
        queryset=ResourceTag.objects.all(),
        serializer=TagSerializer,
        allow_null=True,
        many=True,
    )
    usage_tag = IntegerInterfacedSerializerField(
        queryset=ResourceTag.objects.all(),
        serializer=TagSerializer,
        allow_null=True,
    )
    dependencies = IntegerInterfacedSerializerField(
        queryset=ResourceType.objects.all(),
        serializer=CompactResourceTypeSerializer,
        allow_null=True,
        many=True,
    )
    has_multiple_consumers = serializers.SerializerMethodField()
    fields_mapping_ = {
        'description': ('description', 'description_en'),
        'has_multiple_consumers': ('has_multiple_consumers', 'idempotent_request'),
        'form_link': ('form_id',),
    }

    class Meta:
        model = ResourceType
        fields = [
            'category',
            'description',
            'form_link',
            'has_editable_tags',
            'has_multiple_consumers',
            'has_supplier_tags',
            'has_tags',
            'id',
            'code',
            'is_enabled',
            'is_important',
            'name',
            'supplier',
            'tags',
            'usage_tag',
            'dependencies',
            'need_monitoring',
        ]

    def get_has_multiple_consumers(self, obj):
        try:
            return obj['has_multiple_consumers'] and not obj['idempotent_request']
        except TypeError:
            return obj.has_multiple_consumers and not obj.idempotent_request


class ResourceTypeFilter(PlanFilterSet):
    service = django_filters.CharFilter(method='filter_service')
    category = CustomModelMultipleChoiceFilter(
        field_name='category',
        queryset=ResourceTypeCategory.objects.all(),
    )
    supplier = CustomModelMultipleChoiceFilter(
        field_name='supplier',
        queryset=Service.objects.all(),
    )

    def filter_service(self, queryset, name, pk_or_slug):
        with_childs = self.data.get('with_childs', '').lower() == 'true'
        services = get_services(pk_or_slug, with_childs)
        if not services:
            raise NotFound(detail=_('Сервис не найден'))

        return queryset.filter(resource__serviceresource__service__in=services)

    class Meta:
        model = ResourceType
        fields = {
            'form_id': ['exact', 'isnull'],
            'is_enabled': ['exact'],
            'is_important': ['exact'],
            'name': ['exact', 'contains'],
            'service': ['exact'],
            'supplier': ['exact', 'in'],
        }


class ResourceTypesView(TvmAccessMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """Список известных типов ресурсов. Используется для отображения фильтров на
    фронте. Тот же интерфейс, что и у ручки ресурсов, но без фильтров и
    паджинации"""
    permission_classes = [TvmAuthenticated]
    serializer_class = TypeSerializer
    filter_class = ResourceTypeFilter
    queryset = (
        ResourceType.objects.select_related('supplier', 'category', 'usage_tag')
        .prefetch_related('consumer_roles', 'supplier_roles', 'tags', 'dependencies')
    )
    http_method_names = ['get']

    @action(methods=['get'], detail=True)
    def roles(self, request, pk=None):
        resource_type = get_object_or_404(queryset=self.queryset, pk=pk)

        service = None
        if 'service' in request.GET:
            try:
                service = Service.objects.alive().get(pk=request.GET['service'])
            except Service.DoesNotExist:
                pass

        supplier_roles = merge_allowed_roles(
            resource_type.supplier_roles, resource_type.supplier_scopes, resource_type.supplier
        )
        consumer_roles = merge_allowed_roles(resource_type.consumer_roles, resource_type.consumer_scopes, service)
        roles = supplier_roles | consumer_roles
        has_all_consumer_scopes = set(resource_type.consumer_scopes.all()) == set(RoleScope.objects.all())

        return Response({
            'consumer_roles': RoleSerializer(consumer_roles, many=True).data,
            'supplier_roles': RoleSerializer(supplier_roles, many=True).data,
            'results': RoleSerializer(roles, many=True).data,
            'count': roles.count(),
            'has_all_consumer_scopes': has_all_consumer_scopes,
        })


class V4ResourceTypesView(DefaultFieldsMixin, ResourceTypesView):
    default_swagger_schema = SwaggerResources

    pagination_class = ABCCursorPagination
    default_fields = [
        'id',
        'is_enabled',
        'name',
        'code',

        'supplier.id',
        'supplier.slug',
    ]
