from rest_framework import serializers, viewsets

from plan.api import base
from plan.api.fields import CustomPrimaryKeyRelatedField
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.mixins import DefaultFieldsMixin, SelectOnlyFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.resources.api.resource_types import TypeSerializer
from plan.resources.models import Resource, ResourceType
from plan.services.models import Service
from plan.swagger import SwaggerResources


class ResourceSerializer(base.AttributesSerializer, base.ModelSerializer):
    attributes = serializers.SerializerMethodField()
    type = TypeSerializer(read_only=True)

    obsolete_id = CustomPrimaryKeyRelatedField(
        queryset=Resource.objects.all()
    )

    class Meta:
        model = Resource
        fields = [
            'id',
            'external_id',
            'type',
            'link',
            'name',
            'parent',
            'attributes',
            'obsolete_id'
        ]


class ResourceFilter(PlanFilterSet):
    type = CustomModelMultipleChoiceFilter(
        field_name='type',
        queryset=ResourceType.objects.all(),
        distinct=False,
    )
    supplier = CustomModelMultipleChoiceFilter(
        field_name='type__supplier',
        queryset=Service.objects.all(),
        distinct=False,
    )

    class Meta:
        model = Resource
        fields = ('type', 'supplier', 'external_id', 'name')


class ResourcesView(base.OrderingMixin, SelectOnlyFieldsMixin, viewsets.ModelViewSet):
    """Ручка ресурсов
    * with_childs - bool. Показывать ресурсы указанного сервиса с учетом
                    ресурсов потомков
    Фильтры:
    * type - только ресурсы конкретных типов. Может быть несколько
    * supplier - только ресурсы из конкретных источников. Может быть несколько
    * search - ищет вхождение строки в полях ресурса. Не затрагивает типы и
               источники
    * page - используется паджинация. Номер страницы
    * page_size - столько ресурсов будет отданно в запросе"""
    serializer_class = ResourceSerializer
    filter_class = ResourceFilter
    search_fields = ('id', 'name', 'external_id', 'attributes')
    http_method_names = ['get']
    queryset = Resource.objects.all()


class V4ResourcesView(DefaultFieldsMixin, ResourcesView, TvmAccessMixin):
    default_swagger_schema = SwaggerResources

    pagination_class = base.ABCCursorPagination
    ordering_fields = ('id',)
    permission_classes = [TvmAuthenticated]

    default_fields = [
        'attributes',
        'external_id',
        'id',
        'name',

        'type.supplier.id',
        'type.supplier.slug',
    ]
