import logging

from django.conf import settings
from rest_framework import exceptions, mixins, permissions, serializers, viewsets
from library.python.vault_client.errors import ClientError as VaultClientError

from plan.resources.exceptions import SupplierError, SupplierDelayingError
from plan.resources.models import Resource, ResourceType, ServiceResource
from plan.resources.suppliers.base import SupplierPlugin
from plan.resources.suppliers.robots import add_role_robot_manager_to_staff
from plan.services.models import Service
from plan.staff.models import Staff
from plan.api.exceptions import ValidationError

log = logging.getLogger(__name__)


class RobotViewPermissions(permissions.IsAuthenticated):
    def has_permission(self, request, view):
        if not super(RobotViewPermissions, self).has_permission(request, view):
            return False

        return request.person.user.has_perm('internal_roles.create_robots')


class RobotRequestSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=True
    )
    robot = serializers.CharField(required=True)
    secret_id = serializers.CharField(required=True)
    owner = serializers.SlugRelatedField(
        queryset=Staff.objects.all(),
        required=False,
        slug_field='login',
        default=None,
    )

    def validate_robot(self, robot_login):
        robot = Staff.objects.filter(login=robot_login).first()
        if robot:
            if not robot.is_robot:
                raise ValidationError(
                    detail='This is not a robot: {}'.format(robot_login),
                    message={
                        'ru': 'Сотрудник с указанным логином не является роботом: {}'.format(robot_login),
                        'en': 'The employee with the specified login is not a robot: {}'.format(robot_login)
                    }
                )
            if robot.is_dismissed:
                raise ValidationError(
                    detail='This robot is dismissed: {}'.format(robot_login),
                    message={
                        'ru': 'Указанный робот уволен: {}'.format(robot_login),
                        'en': 'Specified robot is dismissed: {}'.format(robot_login)
                    }
                )
        return robot_login


class RobotView(mixins.CreateModelMixin, viewsets.GenericViewSet):
    permission_classes = (RobotViewPermissions,)
    serializer_class = RobotRequestSerializer

    def perform_create(self, serializer):
        resource_type = ResourceType.objects.get(code='staff-robot')
        supplier_plugin = SupplierPlugin.get_plugin_class(resource_type.supplier_plugin)()
        robot_login = serializer.validated_data['robot']
        service = serializer.validated_data['service']
        secret_id = serializer.validated_data['secret_id']
        owner = serializer.validated_data['owner']
        resource = Resource.objects.filter(type__code='staff-robot', name=robot_login).first()
        if resource is not None:
            service_resources = ServiceResource.objects.filter(resource=resource, state__in=ServiceResource.ALIVE_STATES)
            if not service_resources.filter(service=service).exists():
                raise serializers.ValidationError('This robot is already created')

            prev_secret_id = resource.attributes['secret_id']['value']
            resource.attributes['secret_id']['value'] = secret_id
            resource.attributes['secret_id']['url'] = settings.VAULT_SECRET_URL.format(secret_id=secret_id)
            resource.save(update_fields=['attributes'])
            try:
                for service_resource in service_resources:
                    supplier_plugin.edit(service_resource, prev_secret_id)
            except (SupplierError, VaultClientError) as e:
                raise exceptions.APIException(getattr(e, 'message', str(e)))

        else:
            resource = Resource.objects.create(
                type=resource_type,
                external_id=robot_login,
                name=robot_login,
                link='{}/{}'.format(settings.STAFF_URL, robot_login),
                attributes={
                    'secret_id': {
                        'value': secret_id,
                        'type': 'link',
                        'url': settings.VAULT_SECRET_URL.format(secret_id=secret_id)
                    }
                },
            )
            service_resource = ServiceResource.objects.create(
                resource=resource,
                service=service,
                type=resource_type,
            )
            service_resource.forced_granting()

            if owner:
                add_role_robot_manager_to_staff.apply_async(args=[service.id, owner.id, robot_login])
            try:
                supplier_plugin.create(service_resource)
            except SupplierDelayingError as e:
                log.info(
                    '%s. The robot has not moved to the status GRANTED: %s',
                    service_resource,
                    getattr(e, 'message', str(e)),
                )
            except (SupplierError, VaultClientError) as e:
                raise exceptions.APIException(getattr(e, 'message', str(e)))
            else:
                service_resource.forced_grant()
            service_resource.save()
