import csv
import hashlib
import logging
from collections import defaultdict
from itertools import groupby
from io import BytesIO

import django_filters
from django import http
from django.conf import settings
from django.contrib.postgres.search import SearchVector
from django.utils.encoding import force_text
from django.utils.translation import ugettext_lazy as _
from drf_yasg.utils import swagger_auto_schema
from rest_framework import (
    serializers,
    viewsets,
    status,
)
from rest_framework.decorators import action
from rest_framework.response import Response

from plan.resources.gdpr_export import build_gdpr_workbook
from plan.common.utils.xls import make_attachment_response
from plan.api import base
from plan.api.exceptions import BadRequest, PermissionDenied, IntegrationError, ValidationError, NotFound
from plan.api.fields import IntegerInterfacedSerializerField, CustomPrimaryKeyRelatedField
from plan.api.filters import (
    NoValidationMultipleChoiceFilter,
    CustomModelMultipleChoiceFilter,
    OrderingFilterBackend,
    PlanFilterSet,
    CustomModelChoiceFilter,
)
from plan.history.mixins import HistoryMixin
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin, SelectOnlyFieldsMixin, UnwrapPaginatorMixinWithSubquery
from plan.api.permissions import TvmAuthenticated
from plan.common.utils.sql import PrefixSearchQuery
from plan.resources import models
from plan.resources import permissions
from plan.resources.api.base import get_services, validate_tag_affiliation, Created
from plan.resources.api.filters import ApprovableByFilter
from plan.resources.api.resource_tags import TagSerializer
from plan.resources.api.resources import ResourceSerializer
from plan.resources.policies import BaseApprovePolicy
from plan.resources.suppliers.base import get_supplier_plugin
from plan.resources.tasks import send_csv_from_view
from plan.resources.constants import MAX_CSV_SIZE
from plan.resources.models import ServiceResource
from plan.services.models import Service
from plan.staff.models import Staff
from plan.swagger import SwaggerFrontend, SwaggerResources

log = logging.getLogger(__name__)


class ApproverSerializer(base.CompactStaffSerializer):
    def to_representation(self, instance):
        if instance.login == settings.ABC_ZOMBIK_LOGIN:
            return None

        return super(ApproverSerializer, self).to_representation(instance)


class ServiceResourceListSerializerNoActions(base.ModelSerializer):
    obsolete_id = CustomPrimaryKeyRelatedField(
        queryset=models.ServiceResource.objects.all()
    )

    service = base.CompactServiceSerializer(read_only=True)
    resource = ResourceSerializer(read_only=True)

    state_display = serializers.SerializerMethodField()
    tags = IntegerInterfacedSerializerField(
        queryset=models.ResourceTag.objects.all(),
        serializer=TagSerializer,
        many=True,
    )
    supplier_tags = IntegerInterfacedSerializerField(
        queryset=models.ResourceTag.objects.all(),
        serializer=TagSerializer,
        many=True,
    )
    fields_mapping_ = {'state_display': ('state',)}

    class Meta:
        model = models.ServiceResource
        fields = [
            'id',
            'request_id',
            'modified_at',
            'obsolete_id',
            'resource',
            'service',
            'state',
            'state_display',
            'supplier_tags',
            'tags',
            'created_at',
            'deprived_at',
            'granted_at',
            'approved_at',
            'has_monitoring',
        ]

    @staticmethod
    def get_state_display(obj):
        try:
            value = obj['state']
            display_value = force_text(dict(ServiceResource.STATES).get(value, value), strings_only=True)
        except TypeError:
            display_value = obj.get_state_display()
        return {
            'ru': display_value,
            'en': display_value,
        }

    def validate(self, attrs):
        if self.instance:
            service = self.instance.service
            resource_type = self.instance.resource.type
        else:
            service = attrs['service']
            resource_type = attrs['resource'].type

        tags = attrs.get('tags', ())
        supplier_tags = attrs.get('supplier_tags', ())

        validate_tag_affiliation(service, resource_type, tags, supplier_tags)
        attrs['type'] = resource_type
        return attrs


class ServiceResourceListSerializer(ServiceResourceListSerializerNoActions):
    actions = serializers.SerializerMethodField()
    fields_mapping_ = {'actions': ('state',)}

    class Meta(ServiceResourceListSerializerNoActions.Meta):
        fields = ServiceResourceListSerializerNoActions.Meta.fields + ['actions']

    def get_actions(self, obj):
        if self.context['request'].query_params.get('no_actions'):
            return []

        person = self.context['request'].person
        actions = []
        if isinstance(obj, dict):
            obj = ServiceResource.objects.select_related(
                'resource', 'service', 'resource__type',
                'type', 'type__category', 'type__supplier',
            ).get(id=obj['id'])

        if obj.state == models.ServiceResource.DEPRIVED:
            return actions

        if not permissions.resource_is_active(obj):
            return actions

        resource_policy = BaseApprovePolicy.get_approve_policy_class(obj)(obj)
        supplier_plugin = get_supplier_plugin(obj.type.supplier_plugin)

        if obj.type.form_back_handler and permissions.can_edit_resource(person, resource_policy):
            actions.append('edit')

        if permissions.can_do_extra_actions(person, obj, supplier_plugin):
            actions.extend(supplier_plugin.specific_actions)

        if permissions.can_approve_resource(person, obj):
            actions.append('approve')

        if permissions.can_delete_resource(person, resource_policy, supplier_plugin):
            actions.append('delete')

        if permissions.can_grant_resource(person, obj):
            actions.append('resource_provide')

        return actions


class ServiceResourceSerializer(ServiceResourceListSerializer):
    approvers = serializers.SerializerMethodField()

    depriver = base.CompactStaffSerializer()
    granter = base.CompactStaffSerializer()
    supplier_approver = ApproverSerializer()
    consumer_approver = ApproverSerializer()

    requester = base.CompactStaffSerializer()

    fields_mapping_ = {'approvers': ('state',)}

    def get_approvers(self, obj):
        supplier_approvers, consumer_approvers = obj.get_approvers()
        return {
            'supplier_approvers': base.CompactStaffSerializer(supplier_approvers,
                                                              many=True).data if supplier_approvers is not None else None,
            'consumer_approvers': base.CompactStaffSerializer(consumer_approvers,
                                                              many=True).data if consumer_approvers is not None else None
        }

    class Meta:
        model = models.ServiceResource
        fields = ServiceResourceListSerializer.Meta.fields + [
            'depriver', 'granter', 'supplier_approver',
            'consumer_approver', 'approvers', 'requester',
        ]


class CreateServiceResourceSerializer(base.ModelSerializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.alive(),
        required=True,
    )
    resource = serializers.PrimaryKeyRelatedField(
        queryset=models.Resource.objects.all(),
        required=True,
    )

    tags = IntegerInterfacedSerializerField(
        queryset=models.ResourceTag.objects.all(),
        serializer=TagSerializer,
        many=True,
        required=False,
    )
    supplier_tags = IntegerInterfacedSerializerField(
        queryset=models.ResourceTag.objects.all(),
        serializer=TagSerializer,
        many=True,
        required=False,
    )

    class Meta:
        model = models.ServiceResource
        fields = ['service', 'resource', 'supplier_tags', 'tags']

    def validate(self, attrs):
        if self.instance:
            service = self.instance.service
            resource_type = self.instance.resource.type
        else:
            service = attrs['service']
            resource_type = attrs['resource'].type

        tags = attrs.get('tags', ())
        supplier_tags = attrs.get('supplier_tags', ())

        validate_tag_affiliation(service, resource_type, tags, supplier_tags)
        attrs['type'] = resource_type
        return attrs


class ResourceSearchVector(SearchVector):
    """
    Поскольку SearchVector('resource__id', 'resource__name', 'resource__external_id', config='simple')
    генерирует вызов to_tsvector с полем resources_serviceresource.resource_id, не получается использовать
    единый индекс для поиска ресурсов. Чтобы это обойти возвращается нужный sql для попадания в индекс.
    """
    def as_sql(self, compiler, connection, function=None, template=None):
        return (
            '''
            to_tsvector(%s::regconfig,
                COALESCE("resources_resource"."id"::varchar(32), %s) || \' \' ||
                COALESCE("resources_resource"."name", %s) || \' \' ||
                COALESCE("resources_resource"."external_id", %s))
            ''',
            ['simple', '', '', ''],
        )


class ServiceResourceFilter(PlanFilterSet):
    search = django_filters.CharFilter(method='filter_search')
    service = django_filters.CharFilter(method='filter_service')
    resource = CustomModelChoiceFilter(
        field_name='resource',
        queryset=models.Resource.objects.all(),
        distinct=False,
    )
    state = NoValidationMultipleChoiceFilter(
        field_name='state',
        choices=models.ServiceResource.STATES,
        distinct=False,
    )
    type = CustomModelMultipleChoiceFilter(
        field_name='type',
        queryset=models.ResourceType.objects.all(),
        distinct=False,
    )
    supplier = CustomModelMultipleChoiceFilter(
        field_name='type__supplier',
        queryset=Service.objects.all(),
        distinct=False,
    )
    category = CustomModelMultipleChoiceFilter(
        field_name='type__category',
        queryset=models.ResourceTypeCategory.objects.all(),
        distinct=False,
    )

    tag = CustomModelMultipleChoiceFilter(
        field_name='tags',
        queryset=models.ResourceTag.objects.all(),
    )

    tags_slug = CustomModelMultipleChoiceFilter(
        field_name='tags__slug',
        queryset=models.ResourceTag.objects.all(),
        to_field_name='slug',
    )

    source_tag = CustomModelMultipleChoiceFilter(
        field_name='supplier_tags',
        queryset=models.ResourceTag.objects.all(),
    )

    usage_tag = CustomModelChoiceFilter(
        field_name='tags',
        queryset=models.ResourceTag.objects.all(),
    )

    requester = CustomModelMultipleChoiceFilter(
        field_name='requester__login',
        to_field_name='login',
        queryset=Staff.objects.all(),
    )

    approvable_by = ApprovableByFilter()

    resource__external_id = django_filters.CharFilter(method='filter_by_resource_attr')
    resource__external_id__in = django_filters.CharFilter(method='filter_by_resource_attr')

    resource__name = django_filters.CharFilter(method='filter_by_resource_attr')
    resource__name__in = django_filters.CharFilter(method='filter_by_resource_attr')
    need_monitoring = base.BooleanFilter(method='filter_by_need_monitoring')

    def filter_by_need_monitoring(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if not value:
            return queryset

        type_ids = [
            obj.id for obj in
            models.ResourceType.objects.filter(
                need_monitoring=(value in ('True', True, 'true', '1'))
            )
        ]
        return queryset.filter(type_id__in=type_ids)

    def filter_by_resource_attr(self, queryset, name, value):
        if value == settings.NULL_VALUE:
            return queryset.none()

        if not value:
            return queryset

        if name.endswith('__in'):
            value = set(value.split(','))

        return queryset.filter(**{name: value})

    class Meta:
        model = models.ServiceResource
        fields = {
            'id': ['exact', 'in', 'lt', 'gt'],
            'request_id': ['exact', 'in'],
            'type': ['exact', 'in'],
            'category': ['exact', 'in'],
            'resource': ['exact', 'in'],
            'service': ['exact', 'in'],
            'state': ['exact', 'in'],
            'supplier': ['exact', 'in'],
            'tag': ['exact', 'in'],
            'source_tag': ['exact', 'in'],
            'usage_tag': ['exact', 'in'],
            'obsolete_id': ['exact', 'in', 'isnull'],
            'created_at': ['exact', 'in', 'lt', 'gt'],
            'modified_at': ['exact', 'in', 'lt', 'gt'],
            'has_monitoring': ['exact', ],
            'need_monitoring': ['exact', ],
        }

    def filter_service(self, queryset, name, pk_or_slug):
        with_childs = self.data.get('with_childs', '').lower() == 'true'
        services = get_services(pk_or_slug, with_childs)
        if not services:
            raise NotFound(detail=_('Сервис не найден'))

        values = [service.id for service in services]
        return queryset.filter(service_id__in=values)

    def filter_search(self, queryset, name, value):
        queryset = queryset.annotate(
            search=ResourceSearchVector('resource__id', 'resource__name', 'resource__external_id', config='simple')
        ).filter(
            search=PrefixSearchQuery(value, config='simple')
        )
        return queryset


class ServiceResourcesView(HistoryMixin, TvmAccessMixin, SelectOnlyFieldsMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """Ручка ресурсов, привязанных к сервисам
    * service - фильтр по сервису и его потомкам. Только один в запросе
    * with_childs - bool. Показывать ресурсы указанного сервиса с учетом
                    ресурсов потомков
    Фильтры:
    * type - только ресурсы конкретных типов. Может быть несколько
    * supplier - только ресурсы из конкретных источников. Может быть несколько
    * search - ищет вхождение строки в полях ресурса. Не затрагивает типы и
               источники
    * page - используется паджинация. Номер страницы
    * page_size - столько ресурсов будет отдано в запросе"""
    permission_classes = [TvmAuthenticated]
    filter_class = ServiceResourceFilter
    filter_backends = (django_filters.rest_framework.DjangoFilterBackend, OrderingFilterBackend)
    http_method_names = ['get', 'post', 'patch', 'delete']
    queryset = models.ServiceResource.objects.all()
    ordering_fields = ('id', 'modified_at', 'resource__type_id')
    pagination_class = base.ABCLargePagination
    list_serializer = ServiceResourceListSerializer

    default_select_fields = (
        'type', 'type__category', 'type__supplier',
    )
    default_only_fields = (
        'type__is_enabled',
        'type__is_important',
        'type__supplier_plugin',
        'type__approve_policy',
        'type__form_back_handler',
        'type__supplier__slug',
        'type__category__slug',
    )

    fields_to_export = [
        'modified_at',
        'state',
        'supplier_tags',
        'tags',
        'has_monitoring',

        'resource.attributes',
        'resource.external_id',
        'resource.name',
        'resource.type.supplier.name',
        'resource.type.name',
        'resource.type.need_monitoring',

        'service.name',
        'service.slug',
        'service.id',
    ]

    def get_serializer_class(self):
        if self.action == 'create':
            return CreateServiceResourceSerializer

        if self.request.query_params.get('export'):
            if self.filter_queryset(self.queryset).count() > MAX_CSV_SIZE:
                raise BadRequest(message={
                    'ru': 'Отправка на почту возможна только для выборок с размером меньше %d' % MAX_CSV_SIZE,
                    'en': 'Sending is only possible for samples with size less than %d' % MAX_CSV_SIZE
                })

            request_params = dict(self.request.query_params.lists())
            request_params.pop('export')
            request_params['fields'] = ','.join(self.fields_to_export)
            send_csv_from_view.apply_async(kwargs={
                'path': '/api/resources/serviceresources/',
                'requester_username': self.request.user.username,
                'params': request_params
            })
            raise Created(message={
                'ru': 'Письмо поставлено в очередь на отправку',
                'en': 'Email was added to the queue'
            })

        if self.action == 'list':
            return self.list_serializer

        return ServiceResourceSerializer

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())

        # ToDo: унести в SelectOnlyFieldsMixin.get_queryset(), если будем использовать во всех ручках
        if not queryset._prefetch_related_lookups and self.only_fields is not None:
            queryset = queryset.nested_values(*self.only_fields)
        page = self.paginate_queryset(queryset)
        if page is not None:
            serializer = self.get_serializer(page, many=True)
            return self.get_paginated_response(serializer.data)

        serializer = self.get_serializer(queryset, many=True)
        return Response(serializer.data)

    def perform_create(self, serializer):
        if not permissions.can_request_resource(
                self.request.person,
                serializer.validated_data['service'],
                serializer.validated_data['resource'].type,
                ):
            raise ValidationError(_('Вы не можете запросить этот ресурс'))

        if (not serializer.validated_data['resource'].type.has_multiple_consumers
                and serializer.validated_data['resource'].serviceresource_set.exists()):
            raise ValidationError(_('Многократное использование ресурсов запрещено'))

        already_bound_resource = models.ServiceResource.objects.filter(
            resource=serializer.validated_data['resource'],
            service=serializer.validated_data['service'],
            state__in=models.ServiceResource.ALIVE_STATES,
        )

        if already_bound_resource.exists():
            raise ValidationError(_('Ресурс уже привязан'))

        instance = serializer.save(requester=self.request.user.staff)
        try:
            instance.approve(self.request.person)
        except PermissionDenied:
            pass

    def perform_update(self, serializer):
        if serializer.validated_data.get('state'):
            instance = self.get_object()
            requester = self.request.person
            prev_obj = models.ServiceResource.objects.get(pk=instance.pk)
            new_state = serializer.validated_data['state']

            if prev_obj.state == models.ServiceResource.REQUESTED and new_state == models.ServiceResource.APPROVED:
                prev_obj.approve(approver=self.request.person, request=self.request)

            elif prev_obj.state == models.ServiceResource.APPROVED and new_state == models.ServiceResource.GRANTED:
                instance.grant(granter=requester, request=self.request)

            else:
                raise ValidationError(_('Эта смена состояния объекта не разрешена'))

            return Response(status=status.HTTP_204_NO_CONTENT)

        return super(ServiceResourcesView, self).perform_update(serializer)

    def get_action_from_request(self, request, pk=None):
        service_resource = self.get_object()
        if service_resource.state != models.ServiceResource.GRANTED:
            raise ValidationError(
                message={
                    'ru': 'Работа с секретами возможна только у выданного ресурса',
                    'en': 'Resource is not granted'
                }
            )

        if not permissions.resource_is_active(service_resource):
            message = 'Редактирование ресурса невозможно. Источник отключен.'
            raise PermissionDenied(message={'ru': message})

        supplier_plugin = get_supplier_plugin(service_resource.type.supplier_plugin)
        if not permissions.can_do_extra_actions(request.person, service_resource, supplier_plugin):
            raise PermissionDenied(
                message={
                    'ru': 'У вас недостаточно прав для просмотра мета-информации',
                    'en': "You don't have sufficient permissions to view meta-info"
                }
            )

        if not service_resource.type.supplier_plugin:
            raise ValidationError(_('У этого типа ресурсов нет мета-информации'))

        action = request.data.get('action')
        if action not in supplier_plugin.specific_actions:
            raise ValidationError(_('У этого типа ресурсов нет такого действия: %s' % action))

        return getattr(supplier_plugin, action)

    @action(methods=['post'], detail=True)
    def actions(self, request, pk=None):
        service_resource = self.get_object()
        action = self.get_action_from_request(request, pk=pk)

        try:
            meta_info = action(service_resource, request)
            return Response({'result': meta_info})
        except ValueError as e:
            raise ValidationError(e)

    @action(methods=['post'], detail=True)
    def secrets(self, request, pk=None):
        """
        Работа с секретами
        TODO: выпилить, когда фронт везде перейдет на actions
        """
        return self.actions(request, pk=pk)

    # пока непонятно нужна она кому-то или нет, поэтому убираю (на виики нет её)
    @swagger_auto_schema(auto_schema=None)
    @action(methods=['get'], url_path='passp-export', detail=False)
    def passp_export(self, request):
        """
            Ручка выдачи агрегированных данных для TVM Паспорта
        """
        tvm_application_type_id = request.GET.get('type')

        qs = (models.ServiceResource.objects
              .filter(state__in=models.ServiceResource.ACTIVE_STATES)
              .select_related('service', 'resource', 'type', 'type__supplier',
                              'type__usage_tag'))

        # все ресурсы, у которых в зависимостях есть TVM приложения
        api_resources = (qs.filter(type__dependencies__id=tvm_application_type_id)
                         .order_by('id')
                         .prefetch_related('supplier_tags', 'tags'))

        # все TVM приложения, сгруппированные по потребителю
        grouped_tvm_apps = groupby(
            (qs.filter(type_id=tvm_application_type_id)
                .order_by('service', 'id')
                .prefetch_related('tags')),
            lambda app: app.service
        )

        apps_by_service = {}

        # группируем приложения внутри одного потребителя по тегам
        for service, apps in grouped_tvm_apps:
            apps_by_tag = defaultdict(set)

            for app in apps:
                for tag in app.tags.all():
                    apps_by_tag[tag].add(app)

            apps_by_service[service] = apps_by_tag

        response = http.HttpResponse(content_type='text/plain')
        writer = csv.writer(response, delimiter='\t', lineterminator='\n')
        written_lines = set()

        for api_resource in api_resources:
            # пропускаем все ресурсы, у которых нет атрибута scopes. они не API ресурсы
            scopes = api_resource.resource.attributes.get('scopes')
            if not isinstance(scopes, list):
                continue
            scopes = ','.join(scopes)

            src = api_resource.type.supplier
            dst = api_resource.service
            src_tags = api_resource.supplier_tags.all()
            dst_tags = api_resource.tags.all()

            src_apps_by_tag = apps_by_service.get(src, {})
            dst_apps_by_tag = apps_by_service.get(dst, {})

            src_apps_by_usage_tag = src_apps_by_tag.get(api_resource.type.usage_tag, set())
            dst_apps_by_usage_tag = dst_apps_by_tag.get(api_resource.type.usage_tag, set())

            for src_tag in src_tags:
                # если у ресурса есть usage_tag, то нам нужны все приложения поставщика, которые им отмечены
                src_apps = src_apps_by_tag.get(src_tag, set()) & src_apps_by_usage_tag

                for src_app in src_apps:
                    for dst_tag in dst_tags:
                        # если у ресурса есть usage_tag, то нам нужны все приложения потребителя, которые им отмечены
                        dst_apps = dst_apps_by_tag.get(dst_tag, set()) & dst_apps_by_usage_tag

                        for dst_app in dst_apps:
                            line = [src.id,
                                    src_app.resource.external_id,
                                    dst.id,
                                    dst_app.resource.external_id,
                                    scopes]

                            line_hash = hashlib.md5(repr(line).encode()).digest()

                            if line_hash in written_lines:
                                continue

                            writer.writerow(line)
                            written_lines.add(line_hash)

        return response

    def destroy(self, request, pk=None):
        """
        Отзыв ресурса у сервиса
        """

        service_resource = self.get_object()

        if service_resource.state in models.ServiceResource.ACTIVE_STATES:
            resource_policy = BaseApprovePolicy.get_approve_policy_class(service_resource)(service_resource)
            supplier_plugin = get_supplier_plugin(service_resource.type.supplier_plugin)

            if not permissions.can_delete_resource(request.person, resource_policy, supplier_plugin):
                raise PermissionDenied(
                    message={
                        'ru': 'Это действие доступно только управляющим сервиса-поставщика '
                              'или сервиса-потребителя ресурса.',
                        'en': 'This action is available only to managers of either '
                              'service-supplier or service-consumer of the resource.'
                    }
                )

            if (
                service_resource.type.code == settings.ROBOT_RESOURCE_TYPE_CODE and
                not (
                    ServiceResource.objects
                    .filter(resource=service_resource.resource, state=ServiceResource.GRANTED)
                    .exclude(pk=service_resource.pk)
                    .exists()
                )
            ):
                raise BadRequest(
                    message={
                        'ru': 'Нельзя разорвать последнюю связь робота с сервисом, '
                              'если он больше не нужен, пожалуйста, увольте его - '
                              'https://wiki.yandex-team.ru/tools/support/zombik/#faq ',
                        'en': 'You cant break the robots last connection to the service, '
                              'if robot is no longer needed, please dismiss it - '
                              'https://wiki.yandex-team.ru/tools/support/zombik/#faq ',
                    }
                )

            if supplier_plugin is not None:
                try:
                    supplier_plugin.delete(service_resource, request)
                except Exception as e:
                    raise IntegrationError(getattr(e, 'message', str(e)))

        elif service_resource.state == models.ServiceResource.REQUESTED:

            if not permissions.can_decline_resource(request.person, service_resource):
                raise PermissionDenied(
                    message={
                        'ru': 'Вы не можете отклонить заявку на этот ресурс.',
                        'en': 'You cannot decline this resource.',
                    }
                )

        elif service_resource.state == models.ServiceResource.OBSOLETE:
            raise BadRequest(message={
                'ru': 'Отзыв устаревших ресурсов запрещен',
                'en': 'Depriving of obsolete resources is prohibited'
            })

        service_resource.deprive(request.person.staff)
        service_resource.save()
        return Response({'deleted': True})


class UnwrapABCLargeCursorPagination(UnwrapPaginatorMixinWithSubquery, base.ABCLargeCursorPagination):
    page_size = 100


class V3ServiceResourcesView(ServiceResourcesView):
    list_serializer = ServiceResourceListSerializerNoActions


class V4ServiceResourcesView(DefaultFieldsMixin, V3ServiceResourcesView):
    """
    Ресурсы, привязанные к сервисам.
    Это именно ручка про связь, ServiceResource, между сервисом (сервисами) и ресурсом.
    """
    default_swagger_schema = SwaggerResources

    _additional_ordering_field = '?'
    pagination_class = UnwrapABCLargeCursorPagination
    ordering_fields = ('id', 'modified_at', )
    default_fields = [
        'id',
        'state',
        'modified_at',

        'resource.attributes',
        'resource.external_id',
        'resource.id',
        'resource.name',

        'service.id',
        'service.slug',
    ]

    def optimize_query(self):
        return 'cursor' not in self.request.query_params

    @property
    def additional_ordering_field(self):
        if self.optimize_query():
            return self._additional_ordering_field

    @property
    def use_unwrap(self):
        return self.optimize_query()

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    def actions(self, *args, **kwargs):
        return super(V4ServiceResourcesView, self).actions(*args, **kwargs)

    @swagger_auto_schema(auto_schema=None)
    @action(methods=['post'], detail=True)
    def secrets(self, *args, ** kwargs):
        return super(V4ServiceResourcesView, self).actions(*args, **kwargs)


class GDRPServiceResourceExportView(TvmAccessMixin, viewsets.GenericViewSet):
    default_swagger_schema = SwaggerFrontend
    permission_classes = [TvmAuthenticated]

    queryset = ServiceResource.objects.filter(
        resource__type__code=settings.GDPR_RESOURCE_TYPE_CODE
    ).granted().select_related('service', 'resource').order_by('id')
    filter_class = ServiceResourceFilter
    filter_backends = (django_filters.rest_framework.DjangoFilterBackend, )

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())

        buffer = BytesIO()
        build_gdpr_workbook(queryset, buffer)
        return make_attachment_response(
            data=buffer.getvalue(),
            filename='gdpr_resources.xlsx',
        )


class InfrastructureTypeSerializer(serializers.Serializer):
    code = serializers.CharField()
    content_type = serializers.CharField()


class InfrastructureSerializer(serializers.Serializer):
    type = InfrastructureTypeSerializer()
    content = serializers.CharField()

    def to_representation(self, instance):
        result = super().to_representation(instance)
        code = result['type']['code']
        if code == 'arcadia-url':
            result['content'] = f'{settings.ARCADIA_HOST}/arc/trunk/arcadia/{result["content"]}/'
        elif code == 'infra-preset':
            result['content'] = f'{settings.INFRA_HOST}/timeline?preset={result["content"]}'
        return result


class InfrastructureFrontendView(TvmAccessMixin, viewsets.GenericViewSet):
    default_swagger_schema = SwaggerFrontend

    permission_classes = [TvmAuthenticated]
    serializer_class = InfrastructureSerializer

    def get_queryset(self):
        return ServiceResource.objects.select_related('resource', 'type').alive().filter(
            type__code__in=settings.INFRASTRUCTURE_RESOURCES_TYPES,
        ).order_by('type_id')

    def filter_queryset(self, queryset):
        service_id = self.request.query_params.get('service_id')
        if not service_id:
            raise BadRequest(message={
                'ru': 'Не передан service_id',
                'en': 'Service_id is not provided',
            })
        return queryset.filter(service_id=service_id)

    def _get_content_for_object(self, obj):
        results = []
        obj_type_params = settings.INFRASTRUCTURE_RESOURCES_TYPES[obj.type.code]
        for code, value_path in obj_type_params.items():
            content = obj
            value_path = value_path.split('.')
            for attr_name in value_path:
                if isinstance(content, dict):
                    content = content.get(attr_name)
                else:
                    content = getattr(content, attr_name, None)
            if content is not None:
                if not isinstance(content, list):
                    content = [content]

                for result in content:
                    results.append({
                        'type': {'code': code, 'content_type': 'url'},
                        'content': result,
                    })

        return results

    def _get_data(self, queryset):
        data = []
        for obj in queryset:
            data.extend(self._get_content_for_object(obj))

        return data

    def list(self, request, *args, **kwargs):
        queryset = self.filter_queryset(self.get_queryset())
        data = self._get_data(queryset)

        serializer = self.serializer_class(data, many=True)

        return Response(
            {
                'next': '',
                'previous': '',
                'results': serializer.data,
            }
        )


class FrontendServiceResourcesView(V4ServiceResourcesView):
    default_swagger_schema = SwaggerFrontend
    MAX_RESOURCES_FOR_ALERTS = 300
    PROVIDER_ALERTS_CATEGORY = 'provider_alerts'
    SOLOMON_PROJECTS_HOST = 'https://solomon.yandex-team.ru/admin/projects'

    @action(methods=['get'], detail=False)
    def create_alerts(self, request):
        """
        Редирект на соломон для создания алертов
        """
        service_id = self.request.GET.get('service')
        if not service_id:
            raise BadRequest(message={
                'ru': 'Указание сервиса обязательно',
                'en': 'Service is required',
            })
        service = Service.objects.get(pk=service_id)

        type_id = self.request.GET.get('type')
        if not type_id:
            raise BadRequest(message={
                'ru': 'Указание типа ресурса обязательно',
                'en': 'Type is required',
            })
        resource_type = models.ResourceType.objects.get(pk=type_id)
        if resource_type.category.slug != self.PROVIDER_ALERTS_CATEGORY:
            raise BadRequest(message={
                'ru': 'Создание алертов возможно только для категории Алерты провайдеров',
                'en': 'Creating alerts is possible only for the Provider Alerts category',
            })
        service_provider, resource_type = resource_type.code.split('_', 1)

        queryset = self.filter_queryset(self.get_queryset().select_related('resource'))
        if queryset.count() > self.MAX_RESOURCES_FOR_ALERTS and 'force' not in self.request.GET:
            raise BadRequest(message={
                'ru': 'Слишком много ресурсов для создания алертов',
                'en': 'Too many resources to create alerts',
            })
        query = '&'.join(
            f'r={service_resource.resource.name},resourceType:{resource_type}'
            for service_resource in queryset
        )

        return http.HttpResponseRedirect(
            redirect_to=f'{self.SOLOMON_PROJECTS_HOST}/{service.slug}/alerts/templates/multipleResources?serviceProviderId={service_provider}&{query}'
        )
