import collections

import xlsxwriter

from plan.services.models import Service


COLUMN_NAMES = (
    'Сервис',
    'Действие',
    'Источник данных или адресат данных',
    'Субъекты данных',
    'Срок хранения',
    'Данные',
    'Цель обработки данных',
)

SEND_ACTION = 'Передает'


class GDPRItem:
    def __init__(self):
        self.result_attributes = collections.defaultdict(
            lambda : collections.defaultdict(
                lambda : collections.defaultdict(set)
            )
        )

    def add_attrs(self, attrs, subject, purpose, store_for):
        self.result_attributes[subject][purpose][store_for].add(attrs)


def get_sources_map(queryset):
    sources_slugs = {
        service_resource.resource.attributes['service_from_slug']
        for service_resource in queryset
    }
    return {
        service.slug: service.name
        for service in Service.objects.filter(
            slug__in=sources_slugs,
        )
    }


def get_gdpr_data(queryset):
    result = collections.defaultdict(GDPRItem)

    sources_map = get_sources_map(queryset)
    for service_resource in queryset:
        actions = []
        attributes = service_resource.resource.attributes
        service_name = service_resource.service.name
        data = attributes['data']
        store_for = 'На время обработки'
        purpose = attributes['purpose']
        if attributes['is_using'] == 'Да':
            actions.append('Использует')
        if attributes['is_storing'] == 'Да':
            actions.append('Хранит')
            store_for = attributes['store_for']
        subject = attributes['subject']

        source_name = sources_map[attributes['service_from_slug']]

        keys = [
            (source_name, SEND_ACTION, service_name,),
        ]
        for action in actions:
            keys.append((service_name, action, source_name))

        for key in keys:
            result[key].add_attrs(
                attrs=data,
                store_for=store_for,
                subject=subject,
                purpose=purpose,
            )

    return result


def build_gdpr_workbook(queryset, buffer):
    data = get_gdpr_data(queryset)
    workbook = xlsxwriter.Workbook(buffer, {'in_memory': True})
    worksheet = workbook.add_worksheet('GDPR')

    bold = workbook.add_format({'bold': True})
    for col_number, col_name in enumerate(COLUMN_NAMES):
        worksheet.write(0, col_number, col_name, bold)

    row_number = 1
    for key, gdpr_item in data.items():
        col_number = 0
        for value in key:
            if value:
                worksheet.write(row_number, col_number, value)
                row_number +=1
            col_number +=1
        for subject, purpose_map in gdpr_item.result_attributes.items():
            worksheet.write(row_number, col_number, subject)
            row_number +=1
            col_number +=1
            for purpose, store_for_map in purpose_map.items():
                for store_for, attrs in store_for_map.items():
                    worksheet.write(row_number, col_number, store_for)
                    row_number += 1
                    for attr in sorted(attrs):
                        worksheet.write(row_number, col_number+1, attr)
                        row_number +=1
                if purpose:
                    worksheet.write(row_number, col_number+2, purpose)
            col_number -= 1
            row_number += 1

    workbook.close()
