import logging
import requests

from django.conf import settings

from plan.resources.importers.base import Plugin
from plan.resources.importers.errors import ResourceImporterError
from plan.services.models import Service
from plan.common.utils.tvm import get_tvm_ticket
from plan.common.utils.http import Session

log = logging.getLogger(__name__)


class ArcadiaPlugin(Plugin):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.session = Session(
            headers={'X-Ya-Service-Ticket': get_tvm_ticket(destination=str(settings.ARCADIA_TVM_ID))}
        )

    def _get_link(self, service_slug):
        return f'{settings.ARCADIA_HOST}/projects/{service_slug}/'

    def fetch(self):
        data = dict()

        for resource in self.download(self.resource_type.import_link):
            service_slug = resource.get('id')

            if not service_slug:
                continue

            service_slug = service_slug.lower()

            if service_slug in data:
                log.warning(f'Got duplicate data for slug {service_slug} from arcadia')
                continue

            data[service_slug] = {
                'id': service_slug,
                'name': resource['name'],
                'link': self._get_link(service_slug),
                'attributes': {'dirs': resource['dirs']},
            }

        log.info('Fetch finished')
        result = []

        services = Service.objects.filter(slug__in=data.keys())
        for service in services:
            result.append({
                'service': service,
                'resources': [data[service.slug.lower()]]
            })

        return result

    def download(self, url):
        log.info('Fetch %s with params', url)

        response = self.session.get(url=url)

        if response.status_code == requests.codes.not_found:
            raise ResourceImporterError(
                'Wrong source url',
                status_code=response.status_code
            )

        if not response.ok:
            raise ResourceImporterError(
                f'Invalid response code: {response.status_code}'
            )

        try:
            response_json = response.json()
        except (ValueError, KeyError):
            raise ResourceImporterError(
                f'Invalid json {url}: {response.content}'
            )

        return response_json['data']
