import logging
import urllib.parse

import requests

from django.conf import settings

from plan.common.utils.http import Session
from plan.resources.constants import HR_RESOURCETYPE_CODE
from plan.resources.importers.errors import ResourceImporterError

from .external_resource_data import ExternalResourceData

logger = logging.getLogger(__name__)


class BotServer(ExternalResourceData):
    REQUIRED_FIELDS = {
        'instance_number',
        'segment4',
        'segment3',
        'segment2',
        'segment1',
    }

    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS | {
        'link'
    }

    @property
    def type_name(self):
        return '{s[segment4]}.{s[segment3]}'.format(s=self)

    @property
    def name(self):
        return '{s[segment2]}/{s[segment1]}'.format(s=self)

    @property
    def link(self):
        return self.get('link', None)

    @property
    def external_id(self):
        return self['instance_number']


class ServiceData(ExternalResourceData):
    REQUIRED_FIELDS = {
        'service',
    }

    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS | {
        'link'
    }

    @property
    def type_name(self):
        return HR_RESOURCETYPE_CODE

    @property
    def name(self):
        return self['service_id']

    @property
    def link(self):
        return None

    @property
    def external_id(self):
        return self['service_id']

    @property
    def is_hr(self):
        return self['is_hr'] == 'Y'


def get_raw_bot_data(service_id):
    handle = '/api/services.php?act=info&planner_id={}'.format(service_id)
    url = urllib.parse.urljoin(settings.BOT_URL, handle)

    with Session() as session:
        response = session.post(
            url=url,
            data={'token': settings.OAUTH_ROBOT_TOKEN},
        )

    if response.status_code == requests.codes.not_found:
        raise ResourceImporterError('Service not found',
                                    status_code=response.status_code)

    if not response.ok:
        raise ResourceImporterError(
            'Invalid response code: {r.status_code}'.format(r=response))

    try:
        doc = response.json()

        if not isinstance(doc['servers'], list):
            raise ValueError
    except (ValueError, KeyError):
        raise ResourceImporterError('Invalid json from bot')

    return doc


def get_associated_data(service_id):
    doc = get_raw_bot_data(service_id)
    servers = []
    type_names = set()
    for data in doc['servers']:
        server = BotServer(data)
        servers.append(server)
        type_names.add(server.type_name)
    return {
        'servers': servers,
        'type_names': type_names,
        'service': ServiceData(doc['service'][0]) if len(doc.get('service', [])) else None,
    }
