import logging
import urllib.parse
import collections

import requests

from django.conf import settings

from plan.common.utils.http import Session
from plan.resources.importers.errors import ResourceImporterError
from plan.resources.importers.external_resource_data import ExternalResourceData

logger = logging.getLogger(__name__)


class ConductorData(ExternalResourceData):
    TYPE_NAME = None
    LINK_PATTERN = None

    @property
    def type_name(self):
        return self.TYPE_NAME

    @property
    def name(self):
        return self['name']

    @property
    def link(self):
        return self.LINK_PATTERN.format(self.name)

    @property
    def external_id(self):
        return self.name


class ConductorProject(ConductorData):
    LINK_PATTERN = 'https://c.yandex-team.ru/projects/{}/'
    REQUIRED_FIELDS = {'name', 'abc_service_id'}
    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS
    TYPE_NAME = 'project'


class ConductorGroup(ConductorData):
    LINK_PATTERN = 'https://c.yandex-team.ru/groups/{}/'
    REQUIRED_FIELDS = {'id', 'name'}
    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS
    TYPE_NAME = 'group'
    EXTERNAL_FIELDS = [
        'id',
        'name',
        'description',
        'export_to_racktables',
    ]

    @property
    def external_id(self):
        return self['id']


class ConductorHost(ConductorData):
    LINK_PATTERN = 'https://c.yandex-team.ru/hosts/{}/'
    REQUIRED_FIELDS = {'id', 'fqdn'}
    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS
    TYPE_NAME = 'host'
    EXTERNAL_FIELDS = [
        'id',
        'fqdn',
        'short_name',
        'description',
        'datacenter_name',
    ]

    @property
    def name(self):
        return self['fqdn']

    @property
    def external_id(self):
        return self['id']


def get_services_projects():
    handle = '/api/projects/?format=json'
    url = urllib.parse.urljoin(settings.CONDUCTOR_URL, handle)
    with Session() as session:
        response = session.get(url)
        if not response.ok:
            raise ResourceImporterError('Invalid response code: {r.status_code}'.format(r=response))

    try:
        external_data = response.json()
    except ValueError:
        raise ResourceImporterError('Invalid json from conductor')

    services_projects = collections.defaultdict(list)
    for data in external_data:
        project = ConductorProject(data)
        service_id = project['abc_service_id']
        if not service_id:
            continue

        services_projects[service_id].append(project)

    return services_projects


def crutch_CONDUCTOR_1470(response, data_cls):
    # https://st.yandex-team.ru/CONDUCTOR-1470
    # убрать, когда будет выполнен
    return (
        issubclass(data_cls, ConductorHost) and
        response.status_code == requests.codes.not_found and
        response.content == b'No groups by projects found'
    )


def _fetch(data_cls, handle_pattern, projects):
    project_names = [project.name for project in projects]
    handle = handle_pattern.format(','.join(project_names),
                                   ','.join(data_cls.EXTERNAL_FIELDS))
    url = urllib.parse.urljoin(settings.CONDUCTOR_URL, handle)

    with Session() as session:
        response = session.get(url)

    if not response.ok:
        if crutch_CONDUCTOR_1470(response, data_cls):
            return []
        raise ResourceImporterError('Invalid response code: {r.status_code}'
                                    .format(r=response))

    try:
        external_data = response.json()
    except ValueError:
        raise ResourceImporterError('Invalid json from conductor')

    return [data_cls(data) for data in external_data]


def get_projects_groups(projects):
    handle_pattern = '/api/projects2groups/{}?fields={}&format=json'
    return _fetch(ConductorGroup, handle_pattern, projects)


def get_projects_hosts(projects):
    handle_pattern = '/api/projects2hosts/{}?fields={}&format=json'
    return _fetch(ConductorHost, handle_pattern, projects)
