import datetime
import logging

import requests

from django.conf import settings

from plan.common.utils.dates import datetime_to_str
from plan.resources.importers.base import Plugin
from plan.resources.importers.errors import ResourceImporterError
from plan.services.models import Service

log = logging.getLogger(__name__)


class CrtPlugin(Plugin):
    ATTRIBUTES = ['issued', 'updated', 'revoked', 'status', 'type', 'ca_name', 'username', 'requester', 'hosts']
    QUERY_PARAMS = {
        'abc_service__isnull': False,
        'page_size': 500,
        'no_meta': True,
        'order_by': 'id',
        'id__gt': 0,
        '_fields': ','.join(['id', 'url', 'serial_number', 'common_name', 'abc_service'] + ATTRIBUTES)
    }
    oauth_token = settings.CRT_TOKEN
    # Только ISSUED и DEAD_STATUSES из
    # https://a.yandex-team.ru/arc/trunk/arcadia/intranet/crt/constants.py?rev=r8354387#L162
    STATUSES = ('issued', 'revoked', 'error', 'expired', 'revoking', 'rejected')

    def fetch_incremental(self, since: datetime.datetime = None):
        params = self.QUERY_PARAMS.copy()
        if since is not None:
            params['updated__gt'] = datetime_to_str(since)
        params['status'] = ','.join(self.STATUSES)
        return self._fetch(self.download(self.resource_type.import_link, params))

    def fetch(self):
        params = self.QUERY_PARAMS.copy()
        params['status'] = 'issued'
        return self._fetch(self.download(self.resource_type.import_link, params))

    def _fetch(self, resources):
        result = {}

        for resource in resources:
            service_id = resource.get('abc_service', {}).get('id')

            if not service_id:
                continue

            if service_id not in result:
                try:
                    service = Service.objects.get(pk=service_id)
                except Service.DoesNotExist:
                    log.warning('Service with unknown id <%s> in url <%s>', service_id, self.resource_type.import_link)
                    continue

                result[service_id] = {
                    'service': service,
                    'resources': [],
                }

            requester = resource.pop('requester', {})  # для совместимости со старой версией атрибутов
            attributes = {k: v for k, v in resource.items() if k in self.ATTRIBUTES}
            attributes['username'] = requester.get('username')
            attributes['requester'] = requester.get('username')

            result[service_id]['resources'].append({
                'id': resource['serial_number'],
                'name': resource['common_name'],
                'url': resource['url'],
                'attributes': attributes,
            })

        log.info('Fetch finished')

        return list(result.values())

    def download(self, url, params):
        while True:
            log.info('Fetch %s with params: %s', url, params)

            response = self.session.get(url=url, params=params)

            if response.status_code == requests.codes.not_found:
                raise ResourceImporterError('Wrong source url',
                                            status_code=response.status_code)

            if not response.ok:
                raise ResourceImporterError(
                    'Invalid response code: {r.status_code}'.format(r=response))

            try:
                doc = response.json()
            except (ValueError, KeyError):
                raise ResourceImporterError('Invalid json <%s>: %s' % (url, response.content))

            content = self.get_content(doc)

            if not content:
                break

            for resource in content:
                yield resource

            if params['id__gt'] == resource['id']:
                # На случай, если в CRT что-то пошло не так и он начал всегда возвращать одну и туже страницу
                break

            params['id__gt'] = resource['id']

    def get_content(self, json):
        """
            Ищем блок с данными в ответе
        """
        if isinstance(json, list):
            return json

        if 'content' in json:
            return json['content']

        if 'results' in json:
            return json['results']

        raise ResourceImporterError('Invalid json: no content block found')
