import logging
from urllib.parse import urljoin

import attr

from django.conf import settings

from plan.api.exceptions import IntegrationError
from plan.common.utils.http import Session
from plan.resources.models import Resource, ResourceType, ServiceResource
from plan.services.models import Service

logger = logging.getLogger(__name__)


PROJECTS_API_URL = urljoin(settings.DISPENSER_URL, 'api/v1/projects/')
PROJECTS_FRONTEND_URL = urljoin(settings.DISPENSER_URL, '/projects/')


@attr.s(frozen=True)
class DispenserProject(object):

    key = attr.ib()
    parent_key = attr.ib()
    name = attr.ib()
    service = attr.ib()

    @property
    def link(self):
        return urljoin(PROJECTS_FRONTEND_URL, self.key)

    @classmethod
    def from_dispenser_data(cls, dispenser_data):
        service_id = dispenser_data.get('abcServiceId')
        if service_id is not None:
            service = Service.objects.get(id=service_id)
        else:
            service = None

        parent_key = dispenser_data.get('parentProjectKey')

        return cls(
            name=dispenser_data['name'],
            key=dispenser_data['key'],
            service=service,
            parent_key=parent_key

        )

    def update_resource_attributes(self, resource):
        updated = False
        for attribute in ('name', 'link'):
            existing_attr = getattr(resource, attribute)
            fetched_attr = getattr(self, attribute)

            if existing_attr != fetched_attr:
                updated = True
                setattr(resource, attribute, fetched_attr)

        if self.parent_key:
            if resource.parent is None or resource.parent.external_id != self.parent_key:
                updated = True
                resource.parent = get_or_create_project_resource(self.parent_key)
        elif resource.parent:
            updated = True
            resource.parent = None

        if updated:
            resource.save()

        return resource

    def get_or_create_resource(self):
        if self.parent_key is not None:
            parent = get_or_create_project_resource(self.parent_key)
        else:
            parent = None

        resource = get_or_create_project_resource(self.key, parent)
        return self.update_resource_attributes(resource)

    def get_or_create_service_resource(self):
        if self.service is None:
            return

        resource = self.get_or_create_resource()

        if ServiceResource.objects.filter(resource=resource).exclude(service=self.service).exists():
            (
                ServiceResource
                .objects
                .filter(resource=resource)
                .exclude(service=self.service)
                .update(state=ServiceResource.DEPRIVED)
            )

        service_resource, created = ServiceResource.objects.get_or_create(
            resource=resource,
            service=self.service,
            defaults={'type_id': resource.type_id},
        )

        if service_resource.state != ServiceResource.GRANTED:
            service_resource.state = ServiceResource.GRANTED
            service_resource.save()

        return service_resource


def get_or_create_project_resource(key, parent=None):
    dispenser_type = ResourceType.objects.get(code='dispenser_project')
    resource, created = Resource.objects.get_or_create(
        type=dispenser_type,
        external_id=key,
        defaults={'parent': parent}
    )

    return resource


def get_projects():
    with Session(oauth_token=settings.OAUTH_ROBOT_TOKEN, headers={'Content-type': 'application/json'}) as session:
        response = session.get(PROJECTS_API_URL)

    if response.ok:
        for dispenser_data in response.json()['result']:
            yield DispenserProject.from_dispenser_data(dispenser_data)

    else:
        raise IntegrationError(
            detail='Failed to reach dispenser, dispenser answered with {}'.format(response.status_code)
        )
