import abc


class ExternalResourceDataError(Exception):
    pass


class ExternalResourceData(dict, metaclass=abc.ABCMeta):
    REQUIRED_FIELDS = set()
    NO_ATTRIBUTE_FIELDS = set()

    def check_data(self, json_data):
        """Проверяет, что во входных данных есть все поля из REQUIRED_FIELDS"""

        data_keys = set(json_data.keys())
        if self.REQUIRED_FIELDS & data_keys != self.REQUIRED_FIELDS:
            raise ExternalResourceDataError(
                'Invalid json: fields {0} does not exists'.format(
                    tuple(self.REQUIRED_FIELDS - data_keys)
                ))

    def __init__(self, json_data):
        super(ExternalResourceData, self).__init__()
        self.check_data(json_data)
        self.update(json_data)

    @property
    def attributes(self):
        attribute_keys = set(self.keys()) - self.NO_ATTRIBUTE_FIELDS
        return {key: self[key] for key in attribute_keys}

    def update_db_resource(self, resource):
        """Обновляет и апдейтит ресурс, если это необходимо"""
        changed = False
        for attr in ['name', 'attributes', 'link']:
            external_field = getattr(self, attr)
            obj_field = getattr(resource, attr)
            if obj_field != external_field:
                setattr(resource, attr, external_field)
                changed = True

        if changed:
            resource.save()

    @abc.abstractproperty
    def type_name(self):
        pass

    @abc.abstractproperty
    def name(self):
        pass

    @abc.abstractproperty
    def link(self):
        pass

    @abc.abstractproperty
    def external_id(self):
        pass
