import logging
import urllib.parse

import attr

from django.conf import settings

from plan.api.exceptions import IntegrationError
from plan.common.utils.http import Session
from plan.resources.importers.base import BaseObject
from plan.services.models import Service

logger = logging.getLogger(__name__)


@attr.s(frozen=True)
class FemidaObject(BaseObject):
    id = attr.ib()

    @property
    def link(self):
        return urllib.parse.urljoin(self.base_url, '/vacancies/publications/{id}/'.format(id=self.id))

    @property
    def resource_params(self):
        return {
            'external_id': self.id,
            'link': self.link,
        }

    @property
    def not_required_params(self):
        return {
            'name': self.name,
        }

    @classmethod
    def from_femida_data(cls, obj, base_url):
        services_ids = [item['id'] for item in obj.get('abc_services', [])]
        services = Service.objects.filter(pk__in=services_ids)
        return cls(
            id=obj['id'],
            base_url=base_url,
            name=obj.get('publication_title'),
            services=services,
        )


def get_all_vacancies(base_url):
    fields = ('id', 'publication_title', 'abc_services')
    params = {
        'page_size': 100,
        '_fields': ','.join(fields),
    }
    url = urllib.parse.urljoin(base_url, '/api/publications/')
    with Session(oauth_token=settings.OAUTH_ROBOT_TOKEN) as session:
        while True:
            response = session.get(url, params=params)

            if not response.ok:
                raise IntegrationError(
                    detail='Femida responded with {}'.format(response.content)
                )

            response = response.json()
            results = response['results']
            for obj in results:
                yield FemidaObject.from_femida_data(obj, base_url)

            url = response.get('next')
            if url is None:
                break
