import logging
from urllib.parse import parse_qsl

import requests

from plan.resources.importers.base import Plugin
from plan.resources.importers.errors import ResourceImporterError
from plan.services.models import Service

log = logging.getLogger(__name__)

MAX_PAGES = 900


class GenericPlugin(Plugin):
    oauth_token = None

    def fetch(self):
        result = {}

        for resource in self.download(self.resource_type.import_link):
            for service_data in resource['services']:
                service_id = service_data['abc_service_id']

                if service_id not in result:
                    try:
                        service = Service.objects.get(pk=service_id)
                    except Service.DoesNotExist:
                        log.warning('Service with unknown id <%s> in url <%s>',
                                    service_id, self.resource_type.import_link)
                        continue

                    result[service_id] = {
                        'service': service,
                        'resources': [],
                    }

                result[service_id]['resources'].append(resource)

        log.info('Fetch finished')

        return list(result.values())

    def download(self, url):
        page = 1

        while page:
            query_params = {
                'page': page,
            }
            resource_url = url

            if '?' in url:
                resource_url, qs = url.split('?', 1)
                query_params.update(dict(parse_qsl(qs)))

            log.info('Fetch %s with %r', resource_url, query_params)

            response = self.session.get(url=resource_url, params=query_params)

            if response.status_code == requests.codes.not_found:
                raise ResourceImporterError('Wrong source url',
                                            status_code=response.status_code)

            if not response.ok:
                raise ResourceImporterError(
                    'Invalid response code: {r.status_code}'.format(r=response))

            try:
                doc = response.json()
            except (ValueError, KeyError):
                raise ResourceImporterError('Invalid json')

            content = self.get_content(doc)

            for resource in content:
                yield resource

            if 'next' in doc.get('meta', {}):
                page = doc['meta']['next']
            elif 'next_page' in doc.get('meta', {}):
                page = doc['meta']['next_page']
            elif len(doc['content']):
                page = page + 1
            else:
                page = None

            if page is not None and page > 900:
                raise ResourceImporterError('Too many pages in handle <%s>' % url)

    def get_content(self, json):
        """
            Ищем блок с данными в ответе
        """
        if 'content' in json:
            return json['content']

        if 'results' in json:
            return json['results']

        raise ResourceImporterError('Invalid json: no content block found')
