import logging
from urllib.parse import urljoin

import attr

from django.conf import settings

from plan.api.exceptions import IntegrationError
from plan.common.utils.http import Session
from plan.resources.importers.base import BaseObject
from plan.services.models import Service

logger = logging.getLogger(__name__)
MAX_REQUESTS = 10000


@attr.s(frozen=True)
class NannyObject(BaseObject):
    id = attr.ib()
    description = attr.ib()

    @property
    def link(self):
        return urljoin(self.base_url, '/ui/#/services/catalog/{id}/'.format(id=self.id))

    @property
    def not_required_params(self):
        return {
            'attributes': {
                'description': self.description
            }
        }

    @classmethod
    def from_api_data(cls, obj, base_url):
        service_id = obj.get('info_attrs', {}).get('content', {}).get('abc_group')
        services = Service.objects.filter(pk=service_id)
        return cls(
            id=obj['_id'],
            base_url=base_url,
            name=obj['_id'],
            services=services,
            description=obj.get('info_attrs', {}).get('content', {}).get('desc')
        )


def get_services(base_url):
    url = urljoin(base_url, '/v2/services/')
    params = {'skip': 0}
    with Session(oauth_token=settings.NANNY_TOKEN) as session:
        for _ in range(MAX_REQUESTS):
            response = session.get(url, params=params)

            if not response.ok:
                raise IntegrationError(
                    detail='Nanny responded with {}'.format(response.content)
                )

            results = response.json()['result']
            if not results:
                return

            for obj in results:
                yield NannyObject.from_api_data(obj, base_url)

            params['skip'] += len(results)

    raise IntegrationError(
        detail='Nanny responded with too many pages'
    )
