import logging
from urllib.parse import urljoin

import attr

from django.conf import settings

from plan.api.exceptions import IntegrationError
from plan.common.utils.http import Session
from plan.resources.importers.base import BaseObject
from plan.services.models import Service

logger = logging.getLogger(__name__)


@attr.s(frozen=True)
class BaseQloudObject(BaseObject):
    @property
    def resource_params(self):
        return {
            'name': self.name,
            'link': self.link
        }


@attr.s(frozen=True)
class QloudApplication(BaseQloudObject):
    id = attr.ib()
    project_name = attr.ib()

    @property
    def link(self):
        return urljoin(
            self.base_url,
            '/projects/{project_name}/{name}'.format(project_name=self.project_name, name=self.name),
        )

    @classmethod
    def from_api_data(cls, obj, base_url):
        service_id = obj['metaInfo'].get('abcId')
        services = Service.objects.filter(pk=service_id)
        return cls(
            id=obj['objectId'],
            name=obj['name'],
            project_name=obj['projectName'],
            services=services,
            base_url=base_url
        )

    @property
    def resource_params(self):
        params = super(QloudApplication, self).resource_params
        params['external_id'] = self.id
        return params


@attr.s(frozen=True)
class QloudProject(BaseQloudObject):
    @property
    def link(self):
        return urljoin(self.base_url, '/projects/{name}'.format(name=self.name))

    @classmethod
    def from_api_data(cls, obj, base_url):
        service_id = obj['metaInfo'].get('abcId')
        services = Service.objects.filter(pk=service_id)
        return cls(
            name=obj['projectName'],
            services=services,
            base_url=base_url
        )


def get_projects(base_url):
    with Session(oauth_token=settings.QLOUD_TOKEN) as session:
        response = session.get(urljoin(base_url, '/api/v1/project/'))

    if not response.ok:
        raise IntegrationError(
            detail='Qloud responded with {}'.format(response.content)
        )

    for obj in response.json():
        yield QloudProject.from_api_data(obj, base_url)


def get_applications(base_url, project):
    with Session(oauth_token=settings.QLOUD_TOKEN) as session:
        response = session.get(base_url.format(project_id=project.name))

    if not response.ok:
        raise IntegrationError(
            detail='Qloud responded with {}'.format(response.content)
        )

    for obj in response.json()['applications']:
        yield QloudApplication.from_api_data(obj, project.base_url)
