import logging
import urllib.parse
import collections

import requests

from django.conf import settings

from plan.common.utils.http import Session
from plan.resources.importers.errors import ResourceImporterError
from plan.resources.importers.external_resource_data import ExternalResourceData

logger = logging.getLogger(__name__)


class RackTablesData(ExternalResourceData):
    REQUIRED_FIELDS = {
        'id',
        'type',
        'href',
        'name',
        'services',
    }
    NO_ATTRIBUTE_FIELDS = REQUIRED_FIELDS

    @property
    def type_name(self):
        return self['type']

    @property
    def name(self):
        return self['name']

    @property
    def link(self):
        return self['href']

    @property
    def external_id(self):
        return self['id']


def get_data():
    handle = '/export/abc-objects.php'
    url = urllib.parse.urljoin(settings.RACKTABLES_URL, handle)

    with Session() as session:
        response = session.get(url)

    if response.status_code == requests.codes.not_found:
        raise ResourceImporterError(
            'Invalid response code: {r.status_code}'
            .format(r=response))

    try:
        external_data = response.json()
    except ValueError:
        raise ResourceImporterError('Invalid json from racktables')

    racktables_data = [RackTablesData(data) for data in external_data]

    service_data = collections.defaultdict(list)
    for data in racktables_data:
        for service_id in data['services']:
            service_data[int(service_id)].append(data)

    return service_data
