import logging
import requests

from django.conf import settings

from plan.resources.importers.base import Plugin
from plan.resources.importers.errors import ResourceImporterError
from plan.services.models import Service, ServiceTag
from plan.resources.models import ServiceResource


log = logging.getLogger(__name__)


TIER_TAGS_MAP = {
    'd': 'tier_g',
    'c': 'tier_v',  # это не опечатка
    'b': 'tier_b',
    'a': 'tier_a',
}


class WardenPlugin(Plugin):
    EXPECTED_ATTRIBUTES = (
        'humanReadableName', 'infraPreset',
        'tier', 'spiChat', 'goalUrl', 'state',
    )
    oauth_token = settings.OAUTH_ROBOT_TOKEN

    def _get_human_readable_name(self, attributes, resource):
        return attributes.get('humanReadableName') or self._get_name(resource)

    def _get_external_id(self, resource):
        name = self._get_name(resource)
        component = resource.get('parentComponentName')
        return f'{component}/{name}' if component else name

    def _get_name(self, resource):
        return resource['name']

    def _get_link(self, resource):
        name = self._get_name(resource)
        component = resource.get('parentComponentName')
        if component:
            link = f'{settings.WARDEN_HOST}/components/{component}/s/{name}/'
        else:
            link = f'{settings.WARDEN_HOST}/components/{name}/'
        return link

    def fetch(self):
        warden_data = dict()

        for resource in self.download(self.resource_type.import_link):
            service_slug = resource.get('abcServiceSlug')

            if not service_slug:
                continue

            service_slug = service_slug.lower()

            if service_slug in warden_data:
                log.warning(f'Got duplicate data for slug {service_slug} from warden')
                continue

            attributes = {}
            for field in self.EXPECTED_ATTRIBUTES:
                value = resource.get(field)
                if value:
                    attributes[field] = value

            warden_data[service_slug] = {
                'id': self._get_external_id(resource),
                'name': self._get_human_readable_name(attributes, resource),
                'link': self._get_link(resource),
                'attributes': attributes,
            }

        log.info('Fetch finished')
        result = []

        services = Service.objects.filter(slug__in=warden_data.keys())
        for service in services:
            result.append({
                'service': service,
                'resources': [warden_data[service.slug.lower()]]
            })

        return result

    def download(self, url):
        log.info('Fetch %s with params', url)

        response = self.session.get(url=url)

        if response.status_code == requests.codes.not_found:
            raise ResourceImporterError(
                'Wrong source url',
                status_code=response.status_code
            )

        if not response.ok:
            raise ResourceImporterError(
                f'Invalid response code: {response.status_code}'
            )

        try:
            response_json = response.json()
        except (ValueError, KeyError):
            raise ResourceImporterError(
                f'Invalid json {url}: {response.content}'
            )

        return response_json['components']

    def sync_state(self):
        tier_tags_objects = {
            tag.slug: tag
            for tag in ServiceTag.objects.filter(
                slug__in=TIER_TAGS_MAP.values()
            )
        }
        sr = (
            ServiceResource.objects
            .alive()
            .select_related('resource', 'service')
            .filter(type__code=settings.WARDEN_RESOURCE_TYPE_CODE)
        )

        actual_tags = {}
        for service_resource in sr:
            tier_tag = service_resource.resource.attributes.get('tier')
            if tier_tag:
                actual_tags[service_resource.service_id] = (TIER_TAGS_MAP[tier_tag.lower()], service_resource.service)

        tier_services = Service.objects.prefetch_related('tags').filter(tags__slug__in=TIER_TAGS_MAP.values())

        services_with_tier_tags = {
            serv.id: {
                tag for tag in serv.tags.all()
                if tag.slug in TIER_TAGS_MAP.values()
            }
            for serv in tier_services
        }

        to_delete = services_with_tier_tags.keys() - actual_tags.keys()
        Service.tags.through.objects.filter(
            service_id__in=to_delete,
            servicetag_id__in=(tag.id for tag in tier_tags_objects.values())
        ).delete()

        to_add = []
        for service_id, (target_tag_slug, service) in actual_tags.items():
            service_tags_slugs = {tag.slug for tag in services_with_tier_tags.get(service_id, [])}
            if target_tag_slug not in service_tags_slugs:
                to_add.append(
                    Service.tags.through(
                        service_id=service_id,
                        servicetag_id=tier_tags_objects[target_tag_slug].id
                    )
                )
            if service_tags_slugs:
                to_remove = [tag for tag in services_with_tier_tags[service_id] if tag.slug != target_tag_slug]
                if to_remove:
                    service.tags.remove(*to_remove)

        if to_add:
            Service.tags.through.objects.bulk_create(to_add)
