import logging

import requests

from django.conf import settings

from plan.resources.importers.base import Plugin
from plan.resources.importers.errors import ResourceImporterError
from plan.services.models import Service

log = logging.getLogger(__name__)


class WikiPlugin(Plugin):
    oauth_token = settings.OAUTH_ROBOT_TOKEN

    def fetch(self):
        log.info('Fetch %s', self.resource_type.import_link)

        response = self.session.get(url=self.resource_type.import_link)

        if response.status_code == requests.codes.not_found:
            raise ResourceImporterError('Wrong source url',
                                        status_code=response.status_code)

        if not response.ok:
            raise ResourceImporterError(
                'Invalid response code: {r.status_code}'.format(r=response))

        try:
            doc = response.json()
        except (ValueError, KeyError):
            raise ResourceImporterError('Invalid json from wiki')

        result = {}
        for row in doc['data']['rows']:

            data = [
                item['raw'].strip() if isinstance(item['raw'], str) else item['raw']
                for item in row
            ]

            try:
                resource_data = self.resource_type.process_import(data)
            except Exception:
                log.warning('Cannot process record %r from <%s>', data, self.resource_type.import_link)
                continue

            if not resource_data:
                continue

            service_id = resource_data.pop('service_id')

            if service_id not in result:
                try:
                    service = Service.objects.get(pk=service_id)
                except Service.DoesNotExist:
                    log.warning('Service with unknown id <%s> in url <%s>',
                                service_id, self.resource_type.import_link)
                    continue

                result[service_id] = {
                    'service': service,
                    'resources': [],
                }

            result[service_id]['resources'].append(resource_data)

        log.info('Fetch finished')

        return list(result.values())
