import copy
import decimal
import logging
from typing import List, Optional, Dict, Tuple
from urllib.parse import urljoin
from collections import OrderedDict
from yp.client import YpClient
import waffle

from django.conf import settings
from django.contrib.auth.models import Permission
from django.contrib.postgres.fields import JSONField
from django.db import models, transaction
from django.template.loader import render_to_string
from django.utils import timezone
from django.utils.translation import ugettext_lazy as _

from django_fsm import FSMField, transition
from closuretree.models import ClosureModel
from plan.common.utils.startrek import create_issue

from plan.resources.suppliers.base import SupplierPlugin
from plan.api.exceptions import PermissionDenied, BadRequest
from plan.common.queryset import NestedValuesMixin
from plan.common.person import Person
from plan.common.utils.http import Session
from plan.common.utils.oauth import get_abc_zombik
from plan.resources.processing import Processor
from plan.resources.policies import APPROVE_POLICY, BaseApprovePolicy
from plan.roles.models import Role, RoleScope
from plan.services.models import Service, ServiceTag
from plan.staff.base import I18nModel
from plan.staff.models import Staff


log = logging.getLogger(__name__)


PLUGINS = (
    ('dummy', 'Dummy'),
    ('generic', 'Generic'),
    ('crt', 'Cert'),
    ('wiki', 'Wiki'),
    ('bot', 'BOT'),
    ('racktables', 'Racktables'),
    ('conductor', 'Conductor'),
    ('dispenser', 'Dispenser'),
    ('warden', 'Warden'),
    ('arcadia', 'Arcadia'),
)

SUPPLIER_PLUGINS = (
    ('bot', 'BOT'),
    ('robots', 'Robots'),
    ('tvm', 'TVM'),
    ('metrika', 'Metrika'),
    ('direct', 'Direct'),
    ('fake_financial', 'FakeFinancial'),
    ('billing_point', 'Billing aggregation point'),
)

FORMS = (
    ('constructor', _('Конструктор форм')),
)


class ResourceTypeCategory(models.Model):

    name = models.CharField(
        max_length=255,
        verbose_name=_('Название'),
    )
    name_en = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name='Название (en)',
    )
    slug = models.SlugField(
        unique=True,
        verbose_name=_('Код'),
    )
    description = models.TextField(
        null=True, blank=True, default=None,
        verbose_name=_('Описание'),
    )

    class Meta:
        verbose_name = _('Категория типов')
        verbose_name_plural = _('Категории типов')

    def __str__(self):
        return 'ResourceTypeCategory {obj.id} ({obj.slug})'.format(obj=self)


class ResourceTagCategory(models.Model):
    name = models.CharField(
        max_length=255,
        verbose_name=_('Название'),
    )
    name_en = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name='Название (en)',
    )
    slug = models.SlugField(
        unique=True,
        verbose_name=_('Код'),
    )
    order = models.PositiveSmallIntegerField(
        verbose_name='Порядок',
        default=0,
    )
    is_hidden = models.BooleanField(
        verbose_name='Скрытая',
        default=False,
    )

    class Meta:
        verbose_name = _('Категория тегов')
        verbose_name_plural = _('Категории тегов')

    def __str__(self):
        return 'ResourceTagCategory {obj.id} ({obj.slug})'.format(obj=self)


class ResourceTag(I18nModel):
    name = models.CharField(
        max_length=255,
        verbose_name=_('Название'),
    )
    name_en = models.CharField(
        max_length=255,
        default='',
        blank=True,
        verbose_name='Название (en)',
    )
    slug = models.SlugField(verbose_name=_('Код'))
    description = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание',
    )

    description_en = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание (en)',
    )

    INTERNAL = 'internal'
    EXTERNAL = 'external'
    type = models.CharField(
        verbose_name=_('Тип'),
        max_length=50,
        choices=(
            (INTERNAL, _('Внутренний')),
            (EXTERNAL, _('Внешний')),
        )
    )
    category = models.ForeignKey(
        ResourceTagCategory,
        verbose_name=_('Категория'),
        null=True, blank=True, default=None,
        on_delete=models.SET_NULL,
    )
    service = models.ForeignKey(
        Service,
        verbose_name=_('Сервис'),
        null=True, blank=True, default=None,
    )

    class Meta:
        verbose_name = _('Тег')
        verbose_name_plural = _('Теги')
        unique_together = ('slug', 'service')

    def __str__(self):
        service = self.service.slug if self.service else '-'
        return 'ResourceTag {obj.id} ({obj.slug}@{service})'.format(obj=self, service=service)


class ResourceType(I18nModel):
    supplier = models.ForeignKey(
        Service,
        verbose_name=_('Поставщик'),
        related_name='supplier',
    )
    name = models.CharField(
        max_length=128,
        verbose_name=_('Имя'),
        help_text=_('Название типа, принятое в источнике')
    )
    description = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание',
    )
    description_en = models.TextField(
        default='',
        blank=True,
        verbose_name='Описание (en)',
    )
    code = models.CharField(
        max_length=100,
        null=True,
        unique=True
    )
    service_tags = models.ManyToManyField(
        ServiceTag,
        related_name='resource_types',
        verbose_name=_('Теги, которыми помечается сервис с этим типом ресурсов'),
        blank=True,
    )

    # поля про настройки синка ресурсов из внешних систем
    import_link = models.URLField(
        help_text=_('Ссылка на список ресурсов'),
        null=True, blank=True, default=None,
    )
    import_plugin = models.CharField(
        verbose_name=_('Плагин'),
        choices=PLUGINS,
        max_length=100,
        default=PLUGINS[0][0],
    )
    import_handler = models.TextField(
        verbose_name=_('Импорт'),
        help_text=_('Код на python, обрабатывающий входящие данные из api поставщика'),
        null=True, blank=True, default=None,
    )
    supplier_plugin = models.CharField(
        verbose_name=_('Плагин поставщика'),
        choices=SUPPLIER_PLUGINS,
        max_length=100,
        default=None,
        null=True, blank=True,
        help_text='Дополнительные действия после запроса ресурса',
    )

    # поля про конструктор форм
    form_id = models.PositiveIntegerField(
        _('id опроса в конструкторе форм'),
        null=True,
        blank=True,
        default=None
    )
    form_type = models.CharField(
        verbose_name=_('Тип формы'),
        choices=FORMS,
        max_length=100,
        default=FORMS[0][0],
    )
    form_handler = models.TextField(
        verbose_name=_('Обработка'),
        help_text=_('Код на python, обрабатывающий входящие данные из формы'),
        null=True, blank=True, default=None,
    )
    form_back_handler = models.TextField(
        verbose_name=_('Обратная обработка'),
        help_text=_('Код на python, формирующий из ресурса поля для формы'),
        null=True, blank=True, default=None,
    )

    # переключалки поведения работы с типом ресурса
    approve_policy = models.CharField(
        max_length=50, choices=APPROVE_POLICY.APPROVE_POLICIES,
        verbose_name=_('Политика подтверждения'), default=APPROVE_POLICY.NO_APPROVE
    )
    has_editable_tags = models.BooleanField(default=False, verbose_name=_('Разрешено редактирование тегов'))
    has_tags = models.BooleanField(default=False, verbose_name=_('Нужны ли теги потребителя'))
    has_supplier_tags = models.BooleanField(default=False, verbose_name=_('Нужны ли теги поставщика'))
    has_multiple_consumers = models.BooleanField(default=False, verbose_name=_('Можно ли потреблять чужой ресурс'))
    is_enabled = models.BooleanField(default=True, verbose_name=_('Активен'))
    is_important = models.BooleanField(default=False, verbose_name=_('Важный'))
    is_immutable = models.BooleanField(
        verbose_name=_('Неизменяемость ресурсов'),
        help_text=_('Устаревание ресурсов вместо редактирования'),
        default=False
    )
    has_automated_grant = models.BooleanField(
        verbose_name=_('Автоматизированная выдача'),
        help_text=_('Может быть автоматизированно выдан при запросе'),
        default=False
    )

    # поля про пермишны заказа ресурсов
    supplier_roles = models.ManyToManyField(
        Role,
        blank=True, default=None,
        verbose_name=_('Роли поставщика, которым разрешено выдавать ресурсы этого типа'),
        related_name='+'
    )
    supplier_scopes = models.ManyToManyField(
        RoleScope,
        blank=True, default=None,
        verbose_name=_('Скоупы поставщика, которым разрешено выдавать ресурсы этого типа'),
        related_name='+'
    )
    owner_roles = models.ManyToManyField(
        Role,
        limit_choices_to={'service': None},
        blank=True, default=None,
        verbose_name=_('Роли владельца, которым разрешено выдавать ресурсы этого типа'),
        related_name='+',
    )
    owner_scopes = models.ManyToManyField(
        RoleScope,
        blank=True, default=None,
        verbose_name=_('Скоупы владельца, которым разрешено выдавать ресурсы этого типа'),
        related_name='+',
    )
    consumer_roles = models.ManyToManyField(
        Role,
        limit_choices_to={'service': None},
        blank=True, default=None,
        verbose_name=_('Роли потребителя, которым разрешено запрашивать ресурсы этого типа'),
        related_name='+'
    )
    consumer_scopes = models.ManyToManyField(
        RoleScope,
        blank=True, default=None,
        verbose_name=_('Скоупы потребителя, которым разрешено запрашивать ресурсы этого типа'),
        related_name='+'
    )
    consumer_permissions = models.ManyToManyField(
        Permission,
        related_name='type_permissions',
        blank=True, default=None,
        verbose_name=_('Пермишны, по которым можно запрашивать ресурсы этого типа')
    )

    # поля про категоризацию
    category = models.ForeignKey(
        ResourceTypeCategory,
        verbose_name=_('Категория'),
        null=True, blank=True, default=None,
        on_delete=models.SET_NULL,
    )
    tags = models.ManyToManyField(
        ResourceTag,
        related_name='type_tags',
        blank=True, default=None,
        verbose_name=_('Теги для саджеста')
    )
    usage_tag = models.ForeignKey(
        ResourceTag,
        verbose_name=_('Тег взаимодействия'),
        help_text=_('Тег для пометки ресурсов, которые взаимодействуют с этим типом (usage)'),
        null=True, blank=True, default=None,
        on_delete=models.SET_NULL,
    )
    dependencies = models.ManyToManyField(
        'self',
        blank=True, default=None,
        verbose_name=_('Созависимые типы ресурсов'),
    )
    idempotent_request = models.BooleanField(default=False, verbose_name=_('Идемпотентный запрос'))
    max_granting_time = models.DurationField(default=timezone.timedelta(days=1))
    need_monitoring = models.BooleanField(default=False)

    class Meta:
        unique_together = ('name', 'supplier',)
        verbose_name = _('Тип ресурса')
        verbose_name_plural = _('Типы ресурсов')

    def __str__(self):
        return 'ResourceType {obj.id} ({obj.name} from {obj.supplier.name})'.format(obj=self)

    def process(self, handler_field, **kwargs):
        handler = getattr(self, handler_field)
        if not handler:
            raise ValueError('Не прописан код обработчика в поле ' + handler_field)

        processor = Processor(handler)
        result = processor.run(**kwargs)
        return result

    def can_be_granted_automated(self, obsolete_id=None):
        if self.code == settings.TVM_RESOURCE_TYPE_CODE:
            return not obsolete_id
        return self.has_automated_grant

    @property
    def form_link(self):
        if self.form_id is None:
            return None

        return urljoin(settings.CONSTRUCTOR_FORM_FRONTEND_URL, 'surveys/{}/'.format(self.form_id))

    @property
    def survey_api_link(self):
        return urljoin(settings.CONSTRUCTOR_FORM_API_URL, 'v1/surveys/{}/'.format(self.form_id))

    @property
    def form_api_link(self):
        return urljoin(settings.CONSTRUCTOR_FORM_API_URL, 'v1/surveys/{}/form/'.format(self.form_id))

    def get_form_metadata(self):
        if not self.form_id:
            return

        try:
            with Session() as session:
                return session.get(self.form_api_link).json()

        except Exception:
            log.exception(
                'Cannot get form metadata: resource type %s, url %s',
                self, self.form_api_link,
            )
            raise ValueError('Cannot fetch form metadata: %s' % self)

    def process_import(self, data):
        return self.process('import_handler', data=data)

    def process_form(self, data, cleaned_data):
        result = self.process(
            'form_handler',
            data=data,
            form_metadata=self.get_form_metadata(),
            cleaned_data=cleaned_data,
        )
        return result

    def process_form_back(self, **kwargs):
        result = self.process(
            'form_back_handler',
            attributes=kwargs,
            form_metadata=self.get_form_metadata(),
        )
        return result


class Resource(ClosureModel):
    RESERVED_FIELDS = ('external_id', 'name', 'link')

    name = models.TextField(
        verbose_name=_('Имя'),
        help_text=_('Наименование ресурса в источнике',),
        db_index=True,
    )
    attributes = JSONField(
        blank=True,
        default=dict,
        verbose_name=_('Атрибуты'),
        help_text=_('Атрибуты ресурса из источника')
    )
    link = models.URLField(
        null=True,
        blank=True,
        help_text=_('Ссылка ресурса в источнике')
    )
    external_id = models.CharField(
        max_length=128,
        verbose_name=_('Внешний идентификатор'),
        help_text=_('Внутренний идентификатор ресурса в источнике'),
        default=None,
        null=True, blank=True,
    )
    answer_id = models.PositiveIntegerField(
        verbose_name=_('Код форм'),
        help_text=_('Идентификатор ответа в конструкторе форм'),
        null=True,
        blank=True,
        default=None,
        db_index=True,
    )
    type = models.ForeignKey(
        ResourceType,
        verbose_name=_('Тип'),
        help_text=_('Тип ресурса')
    )
    parent = models.ForeignKey(
        'self',
        null=True,
        blank=True,
        verbose_name=_('Предок'),
        help_text=_('Иерархический предок ресурса'),
        related_name='children',
    )
    obsolete = models.ForeignKey(
        'self',
        null=True, blank=True, default=None,
        verbose_name=_('Устаревший'),
        on_delete=models.SET_NULL,
        related_name='obsolete_resource',
    )
    supplier_response = JSONField(
        blank=True,
        default=dict,
        verbose_name=_('Ответ поставщика'),
        help_text=_('Что нам ответил апи поставщика при запросе ресурса')
    )

    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name=_('Время создания'),
        blank=True,
        null=True,
        db_index=True
    )

    provider_hash = models.CharField(
        db_index=True, max_length=64,
        null=True, blank=True
    )

    class Meta:
        verbose_name = _('Ресурс')
        verbose_name_plural = _('Ресурсы')
        unique_together = [('type', 'external_id', 'obsolete'), ('type', 'answer_id', 'obsolete')]

    def __str__(self):
        return 'Resource {obj.id} ({obj.name})'.format(obj=self)


class ServiceResourceQuerySet(NestedValuesMixin, models.QuerySet):
    def active(self):
        return self.filter(type__is_enabled=True)

    def approvable_by(self, person):
        # ToDo: есть мнение, что это совсем не то, что тут нужно
        # ToDo: переделать в задаче ABC-10720
        if person.user.is_superuser:
            return self
        services = person.get_own_responsible_services(with_descendants=True)
        return self.filter(type__supplier__in=services)

    def granted(self):
        return self.filter(state=ServiceResource.GRANTED)

    def forced_grant(self):
        return self.update(state=ServiceResource.GRANTED, granter=get_abc_zombik(), granted_at=timezone.now())

    def forced_deprive(self):
        return self.update(state=ServiceResource.DEPRIVED, depriver=get_abc_zombik(), deprived_at=timezone.now())

    def alive(self):
        return self.filter(state__in=ServiceResource.ALIVE_STATES)

    def update(self, *args, **kwargs):
        kwargs['modified_at'] = timezone.now()
        return super().update(*args, **kwargs)


class ServiceResourceManager(models.Manager.from_queryset(ServiceResourceQuerySet)):

    def create_granted(self, **params):
        service_resource = self.create(
            state=ServiceResource.GRANTED,
            granter=get_abc_zombik(),
            granted_at=timezone.now(),
            **params,
        )
        if not service_resource.does_other_resources_of_same_type_exists_on_service():
            service_resource._add_service_tags()
        return service_resource


class ServiceResource(models.Model):

    objects = ServiceResourceManager()

    MAX_GRANT_RETRIES = 5

    REQUESTED = 'requested'
    APPROVED = 'approved'
    GRANTING = 'granting'
    GRANTED = 'granted'
    DEPRIVING = 'depriving'
    DEPRIVED = 'deprived'
    OBSOLETE = 'obsolete'
    ERROR = 'error'
    STATES = (
        (REQUESTED, 'Запрошен'),
        (APPROVED, 'Подтвержден'),
        (GRANTING, 'Выдается'),
        (GRANTED, 'Выдан'),
        (DEPRIVING, 'Отзывается'),
        (DEPRIVED, 'Отозван'),
        (OBSOLETE, 'Устарел'),
        (ERROR, 'Ошибка')
    )
    ALIVE_STATES = [REQUESTED, APPROVED, GRANTING, GRANTED, DEPRIVING]
    ACTIVE_STATES = [DEPRIVING, GRANTED]
    ALMOST_ACTIVE_STATES = [GRANTING, GRANTED]
    GRANT_IN_PROCESS = [APPROVED, GRANTING]
    STATES_TO_CHANGE_SERVICE_TAGS = [GRANTED, DEPRIVED, OBSOLETE]
    BAD_STATES = [DEPRIVING, DEPRIVED, OBSOLETE, ERROR]

    state = FSMField(
        verbose_name=_('Статус'),
        choices=STATES,
        default=REQUESTED,
        db_index=True,
    )

    attributes = JSONField(
        blank=True,
        null=True,
        default=dict,
        verbose_name=_('Атрибуты'),
    )
    resource = models.ForeignKey(
        Resource,
        verbose_name=_('Ресурс'),
    )
    service = models.ForeignKey(
        Service,
        verbose_name=_('Сервис'),
    )
    request_id = models.PositiveIntegerField(
        verbose_name=_('Код заказа'),
        help_text=_('Идентификатор ответа в конструкторе форм'),
        null=True, blank=True, default=None, db_index=True,
    )
    requester = models.ForeignKey(
        Staff,
        related_name='service_resources',
        verbose_name=_('Запросивший'),
        null=True, blank=True, default=None,
    )

    consumer_approver = models.ForeignKey(
        Staff,
        related_name='consumer_approved_resources',
        verbose_name=_('Подтвердивший со стороны потребителя'),
        null=True, blank=True, default=None
    )
    supplier_approver = models.ForeignKey(
        Staff,
        related_name='supplier_approved_resources',
        verbose_name=_('Подтвердивший со стороны поставщика'),
        null=True, blank=True, default=None
    )
    depriver = models.ForeignKey(
        Staff,
        related_name='deprived_resources',
        verbose_name=_('Отозвавший'),
        null=True, blank=True, default=None
    )
    granter = models.ForeignKey(
        Staff,
        related_name='approved_resources',
        verbose_name=_('Выдавший'),
        null=True, blank=True, default=None
    )
    grant_retries = models.SmallIntegerField(verbose_name=_('Осталось попыток выдачи'), default=MAX_GRANT_RETRIES)

    created_at = models.DateTimeField(
        auto_now_add=True,
        verbose_name=_('Время создания'),
        blank=True,
        null=True,
        db_index=True
    )

    deprived_at = models.DateTimeField(blank=True, null=True, default=None)
    granted_at = models.DateTimeField(blank=True, null=True, default=None)
    approved_at = models.DateTimeField(blank=True, null=True, default=None)
    modified_at = models.DateTimeField(auto_now=True, db_index=True)

    tags = models.ManyToManyField(
        ResourceTag,
        related_name='consumer_tags',
        blank=True, default=None,
        verbose_name=_('Теги от потребителя ресурса')
    )
    supplier_tags = models.ManyToManyField(
        ResourceTag,
        related_name='supplier_tags',
        blank=True, default=None,
        verbose_name=_('Теги от поставщика ресурса')
    )
    obsolete = models.ForeignKey(
        'self',
        null=True, blank=True, default=None,
        verbose_name=_('Устаревший'),
        on_delete=models.SET_NULL,
        related_name='obsolete_service_resource',
    )
    type = models.ForeignKey(
        ResourceType,
        verbose_name=_('Тип'),
        help_text=_('Тип ресурса'),
        db_index=True,
        related_name='serviceresources',
    )
    has_monitoring = models.BooleanField(default=False)

    class Meta:
        verbose_name = _('Ресурс сервиса')
        verbose_name_plural = _('Ресурсы сервисов')
        index_together = [('service', 'id')]

    def __str__(self):
        return 'ServiceResource {obj.id} ({obj.service.name} - {obj.resource.name} - {obj.state})'.format(obj=self)

    def get_approvers(self):
        return BaseApprovePolicy.get_approve_policy_class(self)(self).get_approvers()

    def _approve_one_side(self, person, approvers, fieldname):
        approver = None
        if not getattr(self, fieldname):
            if approvers is None:
                approver = get_abc_zombik()
            elif person.user.is_superuser or approvers.filter(pk=person.staff.pk).exists():
                approver = person.staff

        if approver is None:
            return False

        setattr(self, fieldname, approver)
        return True

    def approve(self, approver, request=None):
        supplier_approvers, consumer_approvers = self.get_approvers()
        supplier_approved = self._approve_one_side(approver, supplier_approvers, 'supplier_approver')
        consumer_approved = self._approve_one_side(approver, consumer_approvers, 'consumer_approver')

        if not (consumer_approved or supplier_approved):
            raise PermissionDenied('Вы не можете подтвердить ресурс чужого поставщика')

        if self.consumer_approver and self.supplier_approver:
            self._approve()
            self.save()
            if self.type.can_be_granted_automated(obsolete_id=self.obsolete_id):
                self.grant(Person(get_abc_zombik()), request=request)
        else:
            self.save(update_fields=['supplier_approver', 'consumer_approver'])

    def grant(self, granter, request=None):
        from plan.resources.tasks import grant_resource
        from plan.resources.permissions import can_grant_resource

        if not self.type.can_be_granted_automated(obsolete_id=self.obsolete_id) and not can_grant_resource(granter, self):
            raise PermissionDenied('Вы не можете выдать ресурс чужого поставщика')

        if self.type.code == settings.YP_RESOURCE_TYPE_CODE and self.resource.attributes.get('donor_slug'):
            if self.service.slug == self.resource.attributes['donor_slug']:
                raise BadRequest(message={
                    'ru': 'Сервис донор и владелец ресурса совпадают',
                    'en': 'Donor and owner of this resource are equal',
                })
            self._process_quota_donation()

        if self.type.code == settings.TVM_RESOURCE_TYPE_CODE and self.obsolete_id is not None:
            # утверждено перемещение tvm ресурса, выполним изменения на стороне tvm
            plugin = SupplierPlugin.get_plugin_class(self.type.supplier_plugin)()
            try:
                response = plugin.move_resource(self, request.tvm_user_ticket)
            except Exception as exc:
                self.state = self.ERROR
                self.resource.supplier_response = str(exc)
                self.save(update_fields=('state',))
                self.resource.save(update_fields=('supplier_response',))
                return
            self.resource.supplier_response = response
            self.resource.save(update_fields=('supplier_response',))

        self.start_granting(granter.staff)
        self.save()
        grant_resource.apply_async(
            args=[self.id], countdown=settings.ABC_DEFAULT_COUNTDOWN
        )

    @transition(field=state, source=REQUESTED, target=APPROVED)
    def _approve(self):
        self.approved_at = timezone.now()

    @transition(field=state, source=APPROVED, target=GRANTING)
    def start_granting(self, granter):
        self.granter = granter

    @transition(field=state, source='*', target=GRANTING)
    def forced_granting(self):
        pass

    @transition(field=state, source='*', target=GRANTED)
    def forced_grant(self, check_tags=False):
        self.granter = get_abc_zombik()
        self._grant_consequences(check_tags=check_tags)

    @transition(field=state, source=GRANTING, target=GRANTED)
    def _grant(self):
        self._grant_consequences()

    @transition(field=state, source=GRANTING, target=ERROR)
    def fail(self):
        assert self.grant_retries <= 0

    @transition(field=state, source='*', target=DEPRIVED)
    def deprive(self, depriver):
        self.depriver = depriver
        self._deprive_consequences()

    @transition(field=state, source='*', target=DEPRIVED)
    def forced_deprive(self, check_tags=False):
        self.depriver = get_abc_zombik()
        self._deprive_consequences(check_tags)

    @transition(field=state, source='*', target=OBSOLETE)
    def replace(self):
        if not self.does_other_resources_of_same_type_exists_on_service():
            self._remove_service_tags()

    def does_other_resources_of_same_type_exists_on_service(self):
        return ServiceResource.objects.filter(
            ~models.Q(resource_id=self.resource_id),
            state=self.GRANTED,
            type_id=self.type_id,
            service_id=self.service_id,
        ).exists()

    def _grant_consequences(self, check_tags=True):
        self.granted_at = timezone.now()
        if check_tags and not self.does_other_resources_of_same_type_exists_on_service():
            self._add_service_tags()

    def _deprive_consequences(self, check_tags=True):
        self.deprived_at = timezone.now()
        if check_tags and not self.does_other_resources_of_same_type_exists_on_service():
            self._remove_service_tags()

    def _add_service_tags(self):
        tags = self.type.service_tags.exclude(pk__in=self.service.tags.all())
        self.service.tags.add(*tags)

    def _remove_service_tags(self):
        self.service.tags.remove(*self.type.service_tags.all())

    def _replace_quota(self, service_resource, new_quota,
                       needed_quota, diff) -> Tuple[Resource, Optional[Resource]]:
        quants = {
            'cpu': decimal.Decimal('0.001'),  # millicores
            'memory': decimal.Decimal('0.000000001'),  # GB -> 1 byte
            'hdd': decimal.Decimal('0.000000001'),  # TB -> 1 byte
            'ssd': decimal.Decimal('0.000000001'),  # TB -> 1 byte
            'ip4': decimal.Decimal('1'),
            'net': decimal.Decimal('1'),
            'io_ssd': decimal.Decimal('0.000001'),
            'io_hdd': decimal.Decimal('0.000001'),
            'gpu_qty': decimal.Decimal('0.001'),  # what is quant for gpu?
        }

        for k in needed_quota:
            needed_quota[k] -= diff.get(k, 0)

        decimal_new_quota = {}
        for key, value in new_quota.items():
            new_value = decimal.Decimal(value)
            quant = quants.get(key, decimal.Decimal('0.001'))
            decimal_new_quota[key] = '0'
            if new_value > quant:
                decimal_new_quota[key] = str(normalize_fraction(new_value))

        new_resource_attributes = copy.deepcopy(service_resource.resource.attributes)
        new_resource_attributes.update(decimal_new_quota)

        if is_empty_yp_resource(new_resource_attributes):
            service_resource.state = ServiceResource.DEPRIVED
            service_resource.save(update_fields=['state'])
            diff_for_ticket = (service_resource.resource, None)
        else:
            new_resource = Resource.objects.create(
                name=create_name_for_yp_resource(new_resource_attributes),
                obsolete=service_resource.resource,
                type_id=service_resource.type_id,
                attributes=new_resource_attributes,
            )
            ServiceResource.objects.create(
                resource=new_resource,
                state=ServiceResource.GRANTED,
                service=service_resource.service,
                type_id=service_resource.type_id,
            )
            service_resource.state = ServiceResource.OBSOLETE
            service_resource.save(update_fields=['state'])
            diff_for_ticket = (service_resource.resource, new_resource)

        if 'donated_to' not in service_resource.resource.attributes:
            service_resource.resource.attributes['donated_to'] = []
        service_resource.resource.attributes['donated_to'].append(self.pk)
        service_resource.resource.save(update_fields=['attributes'])

        return diff_for_ticket

    def _get_free_quota(self, existing_service_resources, processable_fields, source):
        available_quota = {}
        for service_resource in existing_service_resources:
            existing_quota = {
                k: decimal.Decimal(str(v).replace(',', '.'))
                for k, v in service_resource.resource.attributes.items() if k in processable_fields
            }
            for field, value in existing_quota.items():
                if field in available_quota:
                    available_quota[field] += value
                else:
                    available_quota[field] = value

        used_quota = {}
        yp_client = YpClient(self.resource.attributes['location'], config={'token': settings.YP_TOKEN})

        response = yp_client.select_objects(
            'account',
            selectors=['/spec', '/status'],
            filter='[/meta/id] = "abc:service:{}"'.format(source.id),
        )
        if len(response) == 0:
            raise BadRequest({
                'ru': 'У сервиса-донора нет квоты',
                'en': 'Donor does not have quota',
            })
        assert len(response) == 1, 'Expected exactly one object in response'
        spec, status = response[0]
        resource_limits = spec['resource_limits']['per_segment']
        current_segment = self.resource.attributes['segment']
        if current_segment not in resource_limits:
            raise BadRequest({
                'ru': 'У сервиса-донора нет выбранного сегмента',
                'en': 'Donor does not have needed segment',
            })
        immediate_resource_usage = status['immediate_resource_usage'].get('per_segment') or {}
        segment_usage = immediate_resource_usage.get(current_segment) or {}
        used_quota['cpu'] = (segment_usage.get('cpu') or {}).get('capacity') or 0
        used_quota['hdd'] = ((segment_usage.get('disk_per_storage_class') or {}).get('hdd') or {}).get('capacity') or 0
        used_quota['io_hdd'] = (
            (segment_usage.get('disk_per_storage_class') or {}).get('hdd') or {}
        ).get('bandwidth') or 0
        used_quota['ssd'] = ((segment_usage.get('disk_per_storage_class') or {}).get('ssd') or {}).get('capacity') or 0
        used_quota['io_ssd'] = (
            (segment_usage.get('disk_per_storage_class') or {}).get('ssd') or {}
        ).get('bandwidth') or 0
        used_quota['memory'] = (segment_usage.get('memory') or {}).get('capacity') or 0
        used_quota['ipv4'] = (segment_usage.get('internet_address') or {}).get('capacity') or 0

        # Got relations from here
        # https://a.yandex-team.ru/arc/trunk/arcadia/yp/client/api/proto/data_model.proto?rev=7284605#L143
        denominators = {
            'cpu': 1000,            # 1 millicore -> core
            'hdd': 1024 ** 4,       # 1 byte -> TB
            'io_hdd': 1024 ** 2,    # 1 byte/s -> Mb/s
            'ssd': 1024 ** 4,       # 1 byte -> TB
            'io_ssd': 1024 ** 2,    # 1 byte/s -> Mb/s
            'memory': 1024 ** 3,    # 1 byte -> GB
            'ipv4': 1,              # as is
        }

        for key, value in used_quota.items():
            used_quota[key] = decimal.Decimal(str(value).replace(',', '.')) / denominators[key]
        free_quota = {}

        for field in available_quota:
            if field in used_quota:
                free_quota[field] = available_quota[field] - used_quota[field]
            else:
                free_quota[field] = available_quota[field]
        return free_quota

    @transaction.atomic()
    def _process_quota_donation(self):
        processable_fields = settings.YP_RESOURCE_KEYS
        needed_quota = {
            k: decimal.Decimal(str(v).replace(',', '.'))
            for k, v in self.resource.attributes.items() if k in processable_fields
        }
        source = Service.objects.get(slug=self.resource.attributes['donor_slug'])

        existing_service_resources = ServiceResource.objects.filter(
            type=self.type,
            service=source,
            state=ServiceResource.GRANTED,
            resource__attributes__location=self.resource.attributes['location'],
            resource__attributes__segment=self.resource.attributes['segment'],
        ).select_related('resource').order_by('pk')

        if waffle.switch_is_active('get_free_quota_from_yp'):
            free_quota = self._get_free_quota(existing_service_resources, processable_fields, source)

            allocation_failed_fields = set()
            for field in needed_quota:
                needed_quota_for_field = needed_quota[field]
                if needed_quota_for_field > 0 and free_quota.get(field, 0) < needed_quota_for_field:
                    allocation_failed_fields.add(field)

            if allocation_failed_fields:
                allocation_messages_ru = []
                allocation_messages_en = []
                for field in list(allocation_failed_fields):
                    free = free_quota.get(field, 0)
                    needed = needed_quota[field]
                    allocation_messages_ru.append(
                        f'Доступно с учётом аллокации: {free} {field}, запрошено {needed} {field}'
                    )
                    allocation_messages_en.append(
                        f'Available due to allocation: {free} {field}, requested {needed} {field}'
                    )
                raise BadRequest({
                    'ru': (
                        'У сервиса-донора не хватает квоты в связи с аллокацией. ' +
                        '\n'.join(allocation_messages_ru)
                    ),
                    'en': (
                        'Donor service does not have enough quota due to its allocation. ' +
                        '\n'.join(allocation_messages_en)
                    ),
                })

        diff_for_ticket = []
        for service_resource in existing_service_resources:
            existing_quota = {
                k: decimal.Decimal(str(v).replace(',', '.'))
                for k, v in service_resource.resource.attributes.items() if k in processable_fields
            }
            new_quota = existing_quota.copy()
            diff = {}
            for field, value in needed_quota.items():
                if not value or not existing_quota.get(field):
                    continue

                if (
                    field == 'gpu_qty' and
                    service_resource.resource.attributes.get('gpu_model') != self.resource.attributes.get('gpu_model')
                ):
                    continue

                new_quota[field] = max(0, existing_quota[field] - value)
                diff[field] = existing_quota[field] - new_quota[field]

            if existing_quota != new_quota:
                diff_piece = self._replace_quota(service_resource, new_quota, needed_quota, diff)
                diff_for_ticket.append(diff_piece)

            if not any(needed_quota.values()):
                break

        if any(needed_quota.values()):
            raise BadRequest({
                'ru': 'У сервиса-донора не хватает квоты',
                'en': 'Donor does not have enough quota',
            })

        self._create_ticket_for_quota_donation(source, diff_for_ticket)

    def _create_ticket_for_quota_donation(self, source: Service,
                                          diff_for_ticket: List[Tuple[Resource, Optional[Resource]]]):
        destination = self.service
        context = {
            'requester': self.requester,
            'source': source,
            'destination': destination,
            'diff': diff_for_ticket,
            'new_resource': self.resource,
        }

        # тут скорее всего будет пятисотка, если трекер не работает
        # но это примерно то, что мы и хотим, так как создавать заявку без тикета нельзя
        create_issue(
            queue=settings.ST_QUEUE_FOR_QUOTA_DONATION,
            description=render_to_string('yp_quota_ticket.txt', context),
            summary=f'Перенос YP квоты в ABC {source.id}:{destination.id}',
            tags=['yp_quota_move', f'yp_quota_move-{source.slug}', f'yp_quota_move-{destination.slug}']
        )


class ServiceResourceCounter(models.Model):
    resource_type = models.ForeignKey(ResourceType)
    service = models.ForeignKey(Service)
    count = models.IntegerField(null=False)

    class Meta:
        unique_together = ('service', 'resource_type')


def normalize_fraction(value: decimal.Decimal) -> decimal.Decimal:
    normalized = value.normalize()
    sign, digit, exponent = normalized.as_tuple()
    return normalized if exponent <= 0 else normalized.quantize(1)


def is_empty_yp_resource(attributes: Dict[str, str]) -> bool:
    is_empty = True
    for key in settings.YP_RESOURCE_KEYS:
        if attributes.get(key) and attributes.get(key) != '0':
            is_empty = False
            break
    return is_empty


# Вынесено из админки
def create_name_for_yp_resource(attributes: Dict[str, str]) -> str:
    name = []
    name_bits = OrderedDict([
        ('loc', 'location'),
        ('seg', 'segment'),
        ('cpu', 'cpu'),
        ('mem', 'memory'),
        ('hdd', 'hdd'),
        ('ssd', 'ssd'),
        ('ip4', 'ip4'),
        ('net', 'net'),
        ('io_ssd', 'io_ssd'),
        ('io_hdd', 'io_hdd'),
        ('gpu', 'gpu_model'),
        ('gpu_q', 'gpu_qty'),
    ])
    for code, field_name in name_bits.items():
        name.append('%s:%s' % (code, attributes.get(field_name, '0')))

    return '-'.join(name)
