from django.conf import settings

from plan.staff.models import Staff
from plan.resources.policies import BaseApprovePolicy


def resource_is_active(service_resource):
    return service_resource.type.is_enabled


def can_request_resource_api(person, service, resourse_type):
    return (
        resourse_type.code not in settings.SYSTEM_TYPE_CODES or
        can_request_resource(person, service, resourse_type)
    )


def can_request_resource(person, service, resource_type):
    return (
        person.is_supplier_agent(resource_type)
        or person.is_consumer_agent(service, resource_type)
    )


def can_approve_resource(person, service_resource):
    if service_resource.state != service_resource.REQUESTED:
        return False

    if person.user.is_superuser:
        return True

    supplier_approvers, consumer_approvers = service_resource.get_approvers()
    if supplier_approvers is None and consumer_approvers is None:
        return False

    supplier_approvers = supplier_approvers or Staff.objects.none()
    consumer_approvers = consumer_approvers or Staff.objects.none()
    return (supplier_approvers | consumer_approvers).filter(pk=person.staff.pk).exists()


def can_do_extra_actions(person, service_resource, supplier_plugin):
    if service_resource.state != service_resource.GRANTED:
        return False

    if supplier_plugin is None:
        return False

    return supplier_plugin.can_do_extra_actions(person, service_resource)


def can_edit_resource(person, resource_policy):
    service_resource = resource_policy.service_resource

    if service_resource.type.form_id is None:
        return False

    if service_resource.state not in service_resource.ALIVE_STATES:
        return False

    return resource_policy.can_edit(person)


def can_grant_resource(person, service_resource):
    if service_resource.state != service_resource.APPROVED:
        return False

    if person.user.is_superuser:
        return True

    if service_resource.type.code == settings.TVM_RESOURCE_TYPE_CODE:
        policy = BaseApprovePolicy.get_approve_policy_class(service_resource)(service_resource)
        approvers = policy.get_consumer_approvers(ignore_already_approved=True)
        if approvers and approvers.filter(pk=person.staff.pk).exists():
            return True

    return person.is_supplier_agent(service_resource.type)


def can_decline_resource(person, service_resource):
    return (
        can_request_resource(person, service_resource.service, service_resource.type)
        or can_approve_resource(person, service_resource)
    )


def can_delete_resource(person, resource_policy, supplier_plugin=None):
    service_resource = resource_policy.service_resource

    if service_resource.state not in service_resource.ALIVE_STATES:
        return False

    if supplier_plugin is None:
        return resource_policy.can_delete(person)

    return supplier_plugin.can_delete_resource(person, service_resource)
