from django.db.models import Q
from django.utils.translation import ugettext_lazy as _
from django.conf import settings

from plan.common.person import Person
from plan.resources import permissions
from plan.roles.models import Role, merge_allowed_roles
from plan.services.models import ServiceMember, Service
from plan.staff.models import Staff


class ApproverCanChangeMixin:

    def _can_modify(self, person: Person) -> bool:
        service = self.service_resource.service
        resource_type = self.service_resource.type

        is_consumer_agent = person.is_consumer_agent(service, resource_type)
        if is_consumer_agent:
            return is_consumer_agent

        approvers_id_list = self.get_supplier_approvers_logins()
        return person.staff.pk in approvers_id_list

    def can_edit(self, person: Person) -> bool:
        return self._can_modify(person)

    def can_delete(self, person: Person) -> bool:
        return self._can_modify(person)


class ConsumerApproverMixin:
    def get_consumer_approvers(self, ignore_already_approved=False):
        if self.service_resource.consumer_approver is not None and not ignore_already_approved:
            return None

        resource_type = self.service_resource.type
        roles = merge_allowed_roles(
            resource_type.consumer_roles, resource_type.consumer_scopes, self.service_resource.service)
        approvers = self.service_resource.service.members.filter(role__in=roles).values_list('staff')
        return Staff.objects.filter(pk__in=approvers)


class BaseApprovePolicy(object):
    def __init__(self, service_resource):
        self.service_resource = service_resource

    def get_supplier_approvers(self):
        return None

    def get_consumer_approvers(self, ignore_already_approved=False):
        return None

    def get_approvers(self):
        if self.service_resource.state != self.service_resource.REQUESTED:
            return None, None
        return self.get_supplier_approvers(), self.get_consumer_approvers()

    @classmethod
    def get_approve_policy_class(cls, service_resource):
        return APPROVE_POLICY.APPROVE_POLICY_CLASS[service_resource.type.approve_policy]

    def can_edit(self, person: Person) -> bool:
        service = self.service_resource.service
        resource_type = self.service_resource.type
        return permissions.can_request_resource(person, service, resource_type)

    def can_delete(self, person: Person) -> bool:
        service = self.service_resource.service
        resource_type = self.service_resource.type
        return permissions.can_request_resource(person, service, resource_type)


class SupplierApprovePolicy(BaseApprovePolicy):
    def get_supplier_approvers(self):
        if self.service_resource.supplier_approver is not None:
            return None
        approvers = self.get_supplier_approvers_logins()
        return Staff.objects.filter(pk__in=approvers)

    def _get_service(self):
        return self.service_resource.type.supplier

    def get_supplier_approvers_logins(self):
        resource_type = self.service_resource.type
        service = self._get_service()
        roles = merge_allowed_roles(resource_type.supplier_roles, resource_type.supplier_scopes, service)
        return ServiceMember.objects.filter(service=service, role__in=roles).values_list('staff', flat=True)


class RelatedServiceApprovePolicy(ApproverCanChangeMixin, SupplierApprovePolicy):
    """
    Политика берет подтверждающих не из supplier у типа ресурса, а из
    сервиса указанного в атрибутах ресурса
    """
    def _get_service(self):
        return Service.objects.get(
            slug=self.service_resource.resource.attributes['service_from_slug']
        )


class SupplierApproveOrOwnerRolePolicy(ApproverCanChangeMixin, SupplierApprovePolicy):

    def _policy_is_applicable(self):
        if (
                self.service_resource.type.code == settings.YP_RESOURCE_TYPE_CODE and
                self.service_resource.resource.attributes.get('scenario') == 'Перераспределение квоты'
        ):
            return True

        return False

    def get_supplier_approvers_logins(self):
        approvers = set(super().get_supplier_approvers_logins())

        if self._policy_is_applicable():
            roles = self.service_resource.type.owner_roles.values_list('id', flat=True)
            services = self.service_resource.service.get_ancestors(include_self=True).values_list('id', flat=True)
            service_role_approvers = ServiceMember.objects.filter(
                role_id__in=roles, service_id__in=services
            ).values_list('staff', flat=True)

            approvers.update(service_role_approvers)

        return approvers


class OwnerApprovePolicy(SupplierApprovePolicy):
    def get_supplier_approvers(self):
        if self.service_resource.supplier_approver is not None:
            return None

        supplier_approvers = super(OwnerApprovePolicy, self).get_supplier_approvers()

        resource_type = self.service_resource.type
        active_resources = self.service_resource.resource.serviceresource_set.select_related('service')
        if resource_type.code == settings.ROBOT_RESOURCE_TYPE_CODE:
            active_resources = active_resources.filter(
                state__in=[
                    self.service_resource.GRANTED,
                    self.service_resource.GRANTING,
                ]
            )
        else:
            active_resources = active_resources.filter(
                state=self.service_resource.GRANTED
            )

        owner_roles = merge_allowed_roles(resource_type.owner_roles, resource_type.owner_scopes)
        owner_approvers = ServiceMember.objects.filter(
            service__in=active_resources.values_list('service', flat=True),
            role__in=owner_roles,
        )
        merged_approvers = Staff.objects.filter(
            Q(pk__in=supplier_approvers.values_list('pk', flat=True)) |
            Q(pk__in=owner_approvers.values_list('staff', flat=True))
        )
        return merged_approvers


class SupplierAndConsumerApprovePolicy(SupplierApprovePolicy):
    def get_consumer_approvers(self, ignore_already_approved=False):
        approvers = self.service_resource.service.members.filter(role__code__in=Role.RESPONSIBLES).values_list('staff')
        return Staff.objects.filter(pk__in=approvers)


class ConsumerOrMaybeSupplierApprovePolicy(ConsumerApproverMixin, SupplierApprovePolicy):

    def get_approvers(self):
        if self.service_resource.state != self.service_resource.REQUESTED:
            return (None, None)
        supplier_approvers, consumer_approvers = self.get_supplier_approvers(), self.get_consumer_approvers()
        if supplier_approvers and not consumer_approvers:
            return supplier_approvers, None
        else:
            return None, consumer_approvers


class SuperuserApprovePolicy(BaseApprovePolicy):
    def get_consumer_approvers(self, ignore_already_approved=False):
        return Staff.objects.none()

    def get_supplier_approvers(self):
        return Staff.objects.none()


class TVMApprovePolicy(ConsumerApproverMixin, SupplierApprovePolicy):
    """
    Политика используется для TVM ресурсов, работает только если это
    перенос ресурса, при изначальном создании - подтверждающий не требуется
    """

    def is_applicable(self):
        return (
            self.service_resource.obsolete_id and
            self.service_resource.obsolete.service != self.service_resource.service
        )

    def _get_service(self):
        return self.service_resource.obsolete.service

    def get_supplier_approvers(self):
        if self.is_applicable():
            return super().get_supplier_approvers()

    def get_consumer_approvers(self, ignore_already_approved=False):
        if self.is_applicable():
            return super().get_consumer_approvers(ignore_already_approved=ignore_already_approved)


class APPROVE_POLICY:
    NO_APPROVE = 'no'
    SUPPLIER = 'supplier'
    SUPPLIER_OR_OWNER_ROLE = 'supplier_or_owner_role'
    RELATED_SERVICE_SUPPLIER = 'related_service_supplier'
    SUPPLIER_AND_CONSUMER = 'supplier_and_consumer'
    CONSUMER_OR_MAYBE_SUPPLIER = 'consumer_or_maybe_supplier'
    OWNER = 'owner'
    SUPERUSER = 'superuser'
    TVM_RESOURCES = 'tvm_resources'

    APPROVE_POLICIES = (
        (NO_APPROVE, _('Подтверждение не требуется')),
        (SUPPLIER, _('Подтверждает поставщик')),
        (SUPPLIER_OR_OWNER_ROLE, _('Подтверждает поставщик или пользователь с ролью владельца')),
        (RELATED_SERVICE_SUPPLIER, _('Подтверждает ответственный из связанного с ресурсом сервиса')),
        (SUPPLIER_AND_CONSUMER, _('Подтверждают поставщик и потребитель')),
        (CONSUMER_OR_MAYBE_SUPPLIER, _('Подтверждает потребитель или поставщик')),
        (OWNER, _('Подтверждает владелец ресурса')),
        (SUPERUSER, _('Подтверждает только суперпользователь')),
        (TVM_RESOURCES, _('Политика подтверждения для TVM ресурсов')),
    )

    APPROVE_POLICY_CLASS = {
        NO_APPROVE: BaseApprovePolicy,
        SUPPLIER: SupplierApprovePolicy,
        SUPPLIER_OR_OWNER_ROLE: SupplierApproveOrOwnerRolePolicy,
        RELATED_SERVICE_SUPPLIER: RelatedServiceApprovePolicy,
        SUPPLIER_AND_CONSUMER: SupplierAndConsumerApprovePolicy,
        CONSUMER_OR_MAYBE_SUPPLIER: ConsumerOrMaybeSupplierApprovePolicy,
        OWNER: OwnerApprovePolicy,
        SUPERUSER: SuperuserApprovePolicy,
        TVM_RESOURCES: TVMApprovePolicy,
    }
