import operator
import pprint
import sys
import types
import logging

from urllib.parse import urljoin
from encodings.idna import Codec
from requests.exceptions import RequestException
from py.code import Source, Traceback

from django.utils.datastructures import MultiValueDict
from django.utils.encoding import force_bytes, force_text
from django.conf import settings

from plan.common.utils.http import Session

log = logging.getLogger(__name__)


class ProcessorError(Exception):
    pass


class ProcessorSyntaxError(ProcessorError):
    """Ошибка, связанная с проблемами в синтаксисе."""
    def __init__(self, exception, line, lineno, local_variables):
        super(ProcessorSyntaxError, self).__init__(exception, line, lineno, local_variables)
        self.exception = exception
        self.line = line
        self.lineno = lineno
        self.locals = local_variables

    def __repr__(self):
        return force_bytes(str(self))

    def __str__(self):
        variables = sorted([(key, value) for key, value in self.locals.items()
                            if not isinstance(value, (types.FunctionType, type))], key=operator.itemgetter(0))

        variables = '\n'.join(['    %(key)s = %(value)s' % {
            'key': force_text(key),
            'value': force_text(pprint.pformat(value))
        } for key, value in variables])
        result = 'Ошибка: %(exception)s\nСтрока %(lineno)d: %(line)s\nПеременные:\n%(variables)s' % {
            'exception': repr(self.exception),
            'lineno': self.lineno,
            'line': self.line,
            'variables': variables
        }
        return result


class Processor(object):
    def __init__(self, code):
        self.code = force_text(code)
        if 'coding:' not in self.code:
            self.code = '\n' + self.code
        self.code = self.code.replace('\r', '').strip()
        self.source = Source(self.code)
        self.code_object = None

    def compile(self):
        if not self.code_object:
            self.code_object = self.source.compile()

    def make_context(self, **kwargs):
        kwargs['codec'] = Codec()
        kwargs['MultiValueDict'] = MultiValueDict

        return kwargs

    def run(self, **kwargs):
        self.compile()
        context = self.make_context(**kwargs)

        try:
            exec(self.code_object, context)
        except Exception:
            _, exception, tb = sys.exc_info()
            traceback = Traceback(tb)
            error_source = traceback.getcrashentry()
            line = force_text(error_source.statement)
            lineno = error_source.lineno
            raise ProcessorSyntaxError(exception, line, lineno, error_source.locals)

        if 'result' not in context:
            raise ProcessorError('No "result" defined in processor')

        return context['result']


def get_data_source_items(source_type, params):
    #  Внимание, в КФ захардкожен возврат не более 50 элементов ABC-13347
    url = urljoin(settings.CONSTRUCTOR_FORM_API_URL, f'v1/data-source/{source_type}/')
    params['page_size'] = 200
    results = []
    try:
        next_url = url
        while next_url:
            with Session() as session:
                response = session.get(next_url, params=params).json()
                results.extend(response['results'])
                next_url = response['next']

    except RequestException:
        raise ValueError('Cannot fetch form source items: %s' % url)
    return {
        item['text']: item['id']
        for item in results
    }
