import logging
from importlib import import_module
import json

import waffle

from plan.idm.adapters import RoleManager, RoleRequestManager
from plan.idm.exceptions import Conflict, NotFound
from plan.idm.manager import idm_manager
from plan.idm.constants import IDM_NOT_RETRIABLE_ROLE_STATES, IDM_RETURNABLE_ROLE_STATES
from plan.resources.exceptions import SupplierError
from plan.resources.permissions import can_request_resource

logger = logging.getLogger(__name__)


def get_supplier_plugin(plugin_class_key):
    supplier_plugin_class = SupplierPlugin.get_plugin_class(plugin_class_key)
    if supplier_plugin_class is None:
        return None
    return supplier_plugin_class()


class SupplierPlugin(object):
    resource_type = None
    specific_actions = []

    @classmethod
    def get_plugin_class(cls, plugin):
        if plugin is None:
            return None

        plugin_class = {
            'bot': 'BOTPlugin',
            'robots': 'RobotsPlugin',
            'tvm': 'TVMPlugin',
            'metrika': 'MetrikaPlugin',
            'direct': 'DirectPlugin',
            'fake_financial': 'FakeFinancialPlugin',
            'billing_point': 'BillingPoint',
        }[plugin]

        mod = import_module('plan.resources.suppliers.%s' % plugin)
        return getattr(mod, plugin_class)

    def can_delete_resource(self, person, service_resource):
        return can_request_resource(person, service_resource.service, service_resource.type)

    def can_do_extra_actions(self, person, service_resource):
        return can_request_resource(person, service_resource.service, service_resource.type)

    def create(self, service_resource):
        '''Метод, который ходит в поставщика и создает ресурс.
           Должен отдавать external_id, который поставщик присваивает ресурсу.
           Должен поднимать SupplierError, если операция не выполнена.
        '''
        raise NotImplementedError

    def edit(self, service_resource, request):
        '''Метод редактирования ресурса.
           Ничего не отдает.
           Должен поднимать SupplierError, если операция не выполнена.
        '''
        raise NotImplementedError

    def delete(self, service_resource, request):
        '''Метод удаления ресурса.
           Ничего не отдает.
           Должен поднимать SupplierError, если операция не выполнена.
        '''
        raise NotImplementedError


class SupplierPluginWithRoleRequest(SupplierPlugin):
    def __init__(self):
        self.idm_manager = idm_manager()

    @staticmethod
    def role_dict_to_key(role: dict, from_api: bool) -> tuple:
        user = role.get('user')
        group = role.get('group')

        # При запросе роли user: `username`, а при получении роли user: {username: `username`}
        if from_api:
            user = (user or {}).get('username')
            group = (group or {}).get('id')
            path = role['node']['value_path']
        else:
            path = role['path']

        return (
            user,
            group,
            path,
            json.dumps(role['fields_data'], sort_keys=True),
        )

    def build_role_data(self, service_resource, **kwargs):
        """
        Метод, который возвращает данные для запроса ролей в виде списка словарей.
        """
        raise NotImplementedError

    def build_roles_data_for_idm_sync(self, service_resource):
        role_data = self.build_role_data(service_resource, for_sync=True)

        for role in role_data:
            role['type'] = 'active'
            for field in ('comment', 'fields_data', 'request_fields'):
                role.pop(field, None)

            if role.get('group') is not None:
                role.pop('group')
                role['ownership'] = 'group'
            else:
                role.pop('user')
                role['ownership'] = 'personal'
                role['parent_type'] = 'absent'

        return role_data

    def _create_one(self, role_data):
        assert role_data  # Заполненность поля должна проверяться в вызывающем коде
        try:
            supplier_response = RoleRequestManager.request(self.idm_manager, **role_data)
        except Conflict:
            filters = {'type': 'returnable'}
            filters.update(role_data)
            filters.pop('request_fields', None)
            if 'fields_data' in filters:
                filters['fields_data'] = json.dumps(filters['fields_data'], sort_keys=True)
            response = RoleManager.get_roles(self.idm_manager, filters=filters)

            if response['objects']:
                return response['objects'][0]['id'], None
            else:
                raise
        else:
            return supplier_response['id'], supplier_response

    def get_external_id(self, service_resource):
        raise NotImplementedError()

    def validate_role(self, manager, role_in_data, role_id):
        return True

    def create(self, service_resource, raise_on_single_error=True):
        role_data = self.build_role_data(service_resource)
        if role_data is None:
            return None, None

        old_ids = service_resource.attributes.get('role_id')
        if old_ids:
            if not isinstance(old_ids, list):
                old_ids = [old_ids]
        else:
            old_ids = []
        role_ids = [None] * len(role_data)
        role_ids[:len(old_ids)] = old_ids

        supplier_responses = [None] * len(role_data)
        role_states = [None] * len(role_data)
        manager = idm_manager()
        for i, role in enumerate(role_data):
            if role is None:  # роли не должно быть
                continue
            if role_ids[i] is not None:  # роль уже есть
                if self.validate_role(manager=manager, role_in_data=role, role_id=role_ids[i]):
                    role_states[i] = 'ok'
                    continue
            try:
                role_id, supplier_response = self._create_one(role)
            except Conflict:
                logger.exception('Could not find conflicting roles on conflict: role_data=%s', role)
            else:
                supplier_responses[i] = supplier_response
                role_ids[i] = role_id
                role_states[i] = 'ok'

        service_resource.attributes['role_id'] = role_ids
        service_resource.save(update_fields=['attributes'])

        failed_roles = [role for (role, role_id) in zip(role_data, role_ids) if (role and not role_id)]
        if failed_roles and raise_on_single_error:
            raise SupplierError('Some IDM roles were not requested')

        return self.get_external_id(service_resource), supplier_responses

    def delete(self, service_resource, request):
        role_ids = service_resource.attributes.get('role_id')
        logger.info(f'Отзыв ролей {role_ids} для ресурса {service_resource}')
        if not role_ids:
            return
        if not isinstance(role_ids, list):
            role_ids = [role_ids]
        for role_id in role_ids:
            if role_id is not None:
                role_data = RoleManager.get_role(self.idm_manager, role_id)
                if role_data.get('state') in IDM_RETURNABLE_ROLE_STATES:
                    RoleManager.deprive(self.idm_manager, role_id)

    def create_missing(self, service_resource):
        expected_role_data = self.build_role_data(service_resource)
        if expected_role_data is None:
            return None  # Вообще никаких ресурсов быть не должно

        supplier_responses = [None] * len(expected_role_data)
        old_supplier_responses = service_resource.resource.supplier_response
        if not isinstance(old_supplier_responses, list):
            old_supplier_responses = [old_supplier_responses]
        supplier_responses[:len(old_supplier_responses)] = old_supplier_responses

        external_ids = [None] * len(expected_role_data)
        current_ids = service_resource.attributes.get('role_id')
        if not isinstance(current_ids, list):
            current_ids = [current_ids]
        external_ids[:len(current_ids)] = current_ids

        for i, external_id, role_data in zip(list(range(len(external_ids))), external_ids, expected_role_data):
            if role_data is None:
                continue

            try:
                if external_id:
                    old_role = RoleManager.get_role(self.idm_manager, external_id)
                    if old_role['is_active']:
                        continue  # Ничего исправлять не нужно, состояние ролей в IDM актуальное
                    if (
                        not waffle.switch_is_active('retry_failed_roles_for_resources') and
                        old_role['state'] in IDM_NOT_RETRIABLE_ROLE_STATES
                    ):
                        continue  # Не ретраим роли, упавшие в ошибку
            except NotFound:
                pass

            # Мы хотим найти какую-нибудь роль с похожими параметрами, выданную не как связанная
            filters = role_data.copy()
            filters.pop('request_fields', None)
            if filters:
                # чистый json в params отправить не получится
                filters['fields_data'] = json.dumps(filters['fields_data'], sort_keys=True)
                filters.update({
                    'type': 'returnable',
                    'parent_type': 'absent',
                })
                response = RoleManager.get_roles(self.idm_manager, filters=filters)
                similar_roles = response['objects']
                if similar_roles:
                    idm_role = similar_roles[0]
                    logger.info(
                        'Changing corresponding IDM role for service_resource %s from %s to %s',
                        service_resource.resource.id,
                        service_resource.resource.external_id,
                        idm_role['id'],
                    )
                    # Роль запрашивать не надо, достаточно лишь актуализировать информацию о роли
                    external_ids[i], supplier_responses[i] = idm_role['id'], idm_role
                    continue

            # ни одной подходящей роли не нашлось, запрашиваем новую
            external_ids[i], supplier_responses[i] = self._create_one(role_data)

        service_resource.attributes['role_id'] = external_ids
        service_resource.save(update_fields=['attributes'])

        return supplier_responses
