import logging
import urllib.parse

from django.conf import settings
from rest_framework import serializers
from requests.exceptions import HTTPError

from plan.common.utils.http import Session
from plan.services.models import Service
from plan.resources.exceptions import SupplierError
from plan.resources.models import ServiceResource
from plan.resources.suppliers.base import SupplierPlugin

logger = logging.getLogger(__name__)


class ChangeServiceSerializer(serializers.Serializer):
    service = serializers.PrimaryKeyRelatedField(queryset=Service.objects.all(), required=True)
    reason = serializers.CharField(required=True)


class BOTPlugin(SupplierPlugin):
    oauth_token = settings.OAUTH_ROBOT_TOKEN
    specific_actions = ['change_service']

    def change_service(self, service_resource, request):
        serializer = ChangeServiceSerializer(data=request.data.get('data'))
        serializer.is_valid(raise_exception=True)

        ServiceResource.objects.create(
            resource=service_resource.resource,
            service=serializer.validated_data['service'],
            state=ServiceResource.GRANTED,
            request_id=service_resource.request_id,
            requester=service_resource.requester,
            type_id=service_resource.type_id,
        )
        service_resource.attributes = {
            'deprive_reason': serializer.validated_data['reason'],
        }
        service_resource.deprive(request.person.staff)
        service_resource.save()

        with Session(oauth_token=self.oauth_token) as session:
            response = session.post(
                urllib.parse.urljoin(settings.BOT_URL, '/adm/rename.php'),
                data={
                    'inv': service_resource.resource.external_id,
                    'planner_id': serializer.validated_data['service']
                }
            )
            try:
                response.raise_for_status()
            except HTTPError as e:
                raise SupplierError(getattr(e, 'message', str(e)))

    def delete(self, service_resource, request):
        pass
