from operator import itemgetter

from django.conf import settings
import logging

from plan.resources.models import ServiceResource

from plan.idm.adapters import RoleManager
from plan.idm.constants import ACTIVE_STATES, DEPRIVED_STATES
from plan.resources.suppliers.metrika import FinancialBasePlugin, SkipRole, _get_current_service_group

logger = logging.getLogger(__name__)


def _get_main_manager(plugin, service_resource):
    manager = service_resource.resource.attributes.get('main_manager')
    if not manager:
        raise SkipRole('main_manager should be specified')
    return manager


def _get_direct_service_fields_data(plugin, service_resource):
    client_id = service_resource.resource.attributes.get('client_id')
    if client_id is None:
        raise SkipRole('client_id should be specified')
    return {'client_id': client_id}


def _get_direct_main_manager_fields_data(plugin, service_resource):
    client_id = service_resource.resource.attributes.get('client_id')
    passport_login = service_resource.resource.attributes.get('main_manager_passport_login')
    if passport_login is None or client_id is None:
        raise SkipRole('client_id and main_manager_passport_login should be specified')
    return {'client_id': client_id, 'passport-login': passport_login}


def _get_main_manager_request_fields(plugin, sevice_resource):
    return {'service': sevice_resource.service.slug}


class DirectPlugin(FinancialBasePlugin):
    obj_slug = 'client_id'

    roles = [
        {
            'system': settings.DIRECT_SYSTEM,
            'path': '/manager_for_client/',
            'group': _get_current_service_group,
            'fields_data': _get_direct_service_fields_data,
        },
        {
            'system': settings.DIRECT_SYSTEM,
            'path': '/main_manager_for_client/',
            'user': _get_main_manager,
            'fields_data': _get_direct_main_manager_fields_data,
            'request_fields': _get_main_manager_request_fields,
        },
    ]

    def revoke_roles_with_predicate(self, service_resource, predicate):
        revoked_roles = []
        last_exc = None
        has_nones = any([role_id is None for role_id in service_resource.attributes.get('role_id', [])])
        for role_id in service_resource.attributes.get('role_id', []):
            if role_id is None:
                continue  # Роль по каким-то причинам не была выдана  (возможно и не должна была)
            try:
                role = RoleManager.get_role(self.idm_manager, role_id)
                if role['state'] in DEPRIVED_STATES:
                    # роль уже отозвана - просто удаляем из атрибутов
                    revoked_roles.append(role_id)
                    continue
                if predicate(role):
                    RoleManager.deprive(self.idm_manager, role_id)
                    revoked_roles.append(role_id)
                else:
                    revoked_roles.append(role_id)
            except Exception as e:
                last_exc = e
                logger.exception('Exception in revoke_only_personal_roles for service_resource %s', service_resource.id)

        if revoked_roles or has_nones:
            service_resource.attributes['role_id'] = [
                x for x in service_resource.attributes['role_id'] if (x not in revoked_roles and x is not None)
            ]
            if not service_resource.attributes['role_id']:
                del service_resource.attributes['role_id']

            service_resource.save(update_fields=['attributes'])

        if last_exc is not None:
            raise last_exc

    def revoke_only_group_roles(self, service_resource):
        return self.revoke_roles_with_predicate(service_resource, itemgetter('group'))

    def revoke_only_personal_roles(self, service_resource):
        return self.revoke_roles_with_predicate(service_resource, itemgetter('user'))

    def delete(self, service_resource, request):
        """
        тут такие варианты
         - у другого сервиса мог остаться ресурс с таким же менеджером
            - тогда отзываем только групповую роль
         - в данном сервисе мог остаться такой же ресурс с/без -nomanager
            - отзываем только персональную роль
        """
        external_id = str(service_resource.resource.external_id)
        client_id = external_id.split('-')[0]
        no_manager = '-nomanager' in external_id

        resources_with_same_client = (
            ServiceResource.objects
            .granted()
            .filter(
                type=service_resource.type,
                service_id=service_resource.service_id,
                resource__external_id=client_id if no_manager else f'{client_id}-nomanager'
            )
            .exclude(pk=service_resource.pk)
        ).exists()

        other_service_resources = (
            ServiceResource.objects
            .granted()
            .filter(resource_id=service_resource.resource_id)
            .exclude(pk=service_resource.pk)
        ).exists()

        if resources_with_same_client:
            if no_manager:
                # существует другой ресурс с таким же клиентом, с менеджером
                # роли отзывать не нужно
                return
            if other_service_resources:
                # есть ресурс в другом сервисе с менеджером
                # а в этом сервиса без менеджера остался
                # роли не отзываем
                return
            # других связей у этого менеджера с клиентом нет
            # отзовем только роль главного менеджера
            # а роль на группу оставим - ведь есть -nomanager ресурс
            self.revoke_only_personal_roles(service_resource)
        else:
            if other_service_resources:
                # не отзываем роль главного менеджера так как она
                # выдана через такой же ресурс в другом сервисе
                self.revoke_only_group_roles(service_resource)
            else:
                super().delete(service_resource, request)

    def validate_role(self, manager, role_in_data, role_id):
        role_in_idm = RoleManager.get_role(manager, role_id)
        active_role = role_in_idm['state'] in ACTIVE_STATES
        system = (role_in_idm['system']['slug'] == role_in_data['system'])
        path = (role_in_idm['node']['value_path'] == role_in_data['path'])
        match_client_id = (role_in_idm['fields_data'] == role_in_data['fields_data'])
        group = (
            role_in_idm['group'] is None if 'group' not in role_in_data
            else role_in_idm['group'] is not None and role_in_idm['group']['id'] == role_in_data['group']
        )
        user = (
            role_in_idm['user'] is None if 'user' not in role_in_data
            else role_in_idm['user'] is not None and role_in_idm['user']['username'] == role_in_data['user']
        )

        return active_role and system and path and match_client_id and group and user
