import logging

from plan.services.models import Service
from plan.staff.models import ServiceScope

from plan.resources.suppliers.base import SupplierPluginWithRoleRequest

logger = logging.getLogger(__name__)


class SkipRole(Exception):
    '''Кидается, если текущую роль по какой-то причине нужно пропустить при выдаче'''


def _get_current_service_group(plugin, service_resource):
    if plugin.scope is None:
        group_id = Service.objects.get(pk=service_resource.service.id).staff_id
    else:
        group_id = ServiceScope.objects.get(service_id=service_resource.service.id, role_scope_id=plugin.scope).staff_id
    return group_id


class FinancialBasePlugin(SupplierPluginWithRoleRequest):
    obj_slug = None
    roles = None
    scope = None

    def _get_value(self, service_resource, source_dict, key, required=True):
        value = source_dict[key] if required else source_dict.get(key)
        if value is None:
            return value
        if callable(value):
            return value(self, service_resource)
        return value

    def build_role_data(self, service_resource, for_sync=False):
        assert self.roles is not None

        role_data = []
        for role in self.roles:
            try:
                data = {
                    'system': self._get_value(service_resource, role, 'system'),
                    'path': self._get_value(service_resource, role, 'path'),
                }
                user = group = None
                if for_sync:
                    # Конкретный логин все равно удаляется из фильтра. Нужно просто отличать личные роли от групповых
                    # Строка '__any__' нигде не используется
                    if 'user' in role:
                        user = '__any__'
                    else:
                        group = '__any__'
                else:
                    # В синке все равно удаляются user и fields_data
                    user = self._get_value(service_resource, role, 'user', required=False)
                    group = self._get_value(service_resource, role, 'group', required=False)

                    data['fields_data'] = self._get_value(service_resource, role, 'fields_data')
                    data['request_fields'] = self._get_value(service_resource, role, 'request_fields', required=False)

                # Мы рассчитываем на xor(user, group)
                assert (user is None) != (group is None), 'Either user or group should be specified'
                if user is not None:
                    data['user'] = user
                elif group is not None:
                    data['group'] = group

            except SkipRole:
                data = None

            role_data.append(data)

        return role_data

    def get_external_id(self, service_resource):
        return service_resource.resource.external_id  # Сохраняется на этапе запроса ресурса


def _get_metrika_fields_data(plugin, service_resource):
    counter_id = service_resource.resource.attributes.get('counter_id')
    if counter_id is None:
        raise SkipRole('counter_id should be specified')
    return {'counter_id': counter_id}


class MetrikaPlugin(FinancialBasePlugin):
    obj_slug = 'counter_id'

    roles = [
        {
            'system': 'metrika',
            'path': '/external_counter_grant/',
            'group': _get_current_service_group,
            'fields_data': _get_metrika_fields_data,
        },
    ]
