import logging

from django.conf import settings
from django.utils.encoding import force_text

from library.python.vault_client import instances
import yenv

from plan.common.utils.http import Session
from plan.resources.exceptions import SupplierError, RobotIsDismissed, SupplierDelayingError
from plan.resources.suppliers.base import SupplierPluginWithRoleRequest
from plan.roles.models import Role
from plan.services.models import ServiceMember, Service
from plan.staff.models import ServiceScope, Staff
from plan.api.idm import actions
from plan.celery_app import app

logger = logging.getLogger(__name__)

READER_ROLE = 'READER'
SECRET_COMMENT = '''
Доступ к секрету выдаётся только на чтение.
Права на чтение секрета имеют люди с ролями "Пользователь роботов" или "Управляющий роботами" в тех ABC-сервисах, к которым привязан робот.
Изменить пароль робота можно через UI ABC. Токены при этом отозваны не будут.

The only grant issued for this secret is READING.
Read access is only granted for those who own "Robots user" or "Robots manager" role in ABC services who this robot is attached to.
You can change the robot's password in the ABC UI. If you do so, tokens of the robot would NOT be invalidated.
'''


def _get_vault_client(login: str = settings.ABC_ROBOTSECRETS_LOGIN):
    cls = instances.Testing
    if yenv.type == 'production':
        cls = instances.Production

    token = settings.DICT_ROBOT_TOKEN[login]
    return cls(authorization=token)


def process_hd_api_response(response, resource):
    result = response.json()['result']

    status = result['status']
    if status == 'done':
        resource.attributes['password_status'] = 'ok'
        resource.attributes['password_updated_at'] = result['lastUpdate']
    else:
        resource.attributes['password_status'] = status
        resource.attributes['operation'] = result['uuid']

    resource.save(update_fields=['attributes'])


class RobotsPlugin(SupplierPluginWithRoleRequest):
    specific_actions = ['reset_password']
    system = 'staff'
    scope = settings.ABC_ROBOTS_MANAGEMENT_SCOPE
    user_scope = settings.ABC_ROBOTS_USERS_SCOPE

    def can_do_anything(self, person, service_resource):
        return person.has_role(
            service_resource.service,
            Role.objects.globalwide().get(code=settings.ABC_ROBOTS_MANAGER_ROLE)
        )

    def can_do_extra_actions(self, person, service_resource):
        return self.can_do_anything(person, service_resource)

    def can_delete_resource(self, person, service_resource):
        return (
            self.can_do_anything(person, service_resource)
            or person.is_supplier_agent(service_resource.type)
        )

    def build_roles_data_for_idm_sync(self, service_resource):
        # Скачиваем все роли Стаффа, потому что адекватный фильтр сейчас придумать нельзя :(
        return [{
            'system': self.system,
        }]

    @staticmethod
    def validate_robot_request(robot):
        if not robot.is_robot:
            message = '{} is not a robot'.format(robot.login)
            logger.error(message)
            raise SupplierError(message)
        if robot.is_dismissed:
            message = '{} is dismissed'.format(robot.login)
            logger.error(message)
            raise RobotIsDismissed(message)

    def get_role_data(self, group_id, role_type, robot_login, robot_id):
        return {
            'comment': 'Выдача ресурса робота {}'.format(robot_login),
            'group': group_id,
            'system': self.system,
            'path': '/robots/{}/{}/'.format(robot_id, role_type),
            'fields_data': {}
        }

    def build_role_data(self, service_resource, **kwargs):
        result = []
        try:
            group = ServiceScope.objects.get(service_id=service_resource.service.id, role_scope__slug=self.scope)
            robot_login = service_resource.resource.external_id
            robot = Staff.objects.get(login=robot_login)
            self.validate_robot_request(robot)
            robot_id = robot.staff_id
        except ServiceScope.DoesNotExist as e:
            raise SupplierDelayingError(
                '%s. Service %s does not have group for robots management scope.',
                getattr(e, 'message', str(e)),
                service_resource.service,
            )
        except Staff.DoesNotExist as e:
            raise SupplierDelayingError(
                '%s. Robot login: %s', getattr(e, 'message', str(e)), robot_login
            )

        result.append(
            self.get_role_data(
                group_id=group.staff_id, robot_id=robot_id,
                robot_login=robot_login, role_type='owner',
            )
        )

        users_group = ServiceScope.objects.filter(
            service_id=service_resource.service.id,
            role_scope__slug=self.user_scope,
        ).first()
        if users_group:
            result.append(
                self.get_role_data(
                    group_id=users_group.staff_id, robot_id=robot_id,
                    robot_login=robot_login, role_type='user',
                )
            )

        return result

    def get_external_id(self, service_resource):
        return service_resource.resource.external_id  # Сохраняется на этапе запроса ресурса

    def create(self, service_resource):
        super(RobotsPlugin, self).create(service_resource, raise_on_single_error=False)

        has_members = ServiceMember.objects.team().filter(
            service=service_resource.service,
            role__scope__slug=settings.ABC_ROBOTS_MANAGEMENT_SCOPE
        ).exists()

        if not has_members:
            raise SupplierDelayingError('Robots manager is missing.')

        vault_client = _get_vault_client()
        for param in (
            {'abc_scope': settings.ABC_ROBOTS_MANAGEMENT_SCOPE},
            {'abc_role_id': settings.ABC_ROBOTS_USERS_ROLE_ID},
        ):
            if not vault_client.add_user_role_to_secret(
                secret_uuid=service_resource.resource.attributes['secret_id']['value'],
                role=READER_ROLE,
                abc_id=service_resource.service.id,
                **param
            ):
                raise SupplierError(f'Didn\'t grant access to robot password to {param}')

        if not vault_client.update_secret(
            secret_uuid=service_resource.resource.attributes['secret_id']['value'],
            comment=SECRET_COMMENT,
        ):
            raise SupplierError('Comment cannot be updated')

        idm_role_ids = service_resource.attributes.get('role_id')
        if idm_role_ids and idm_role_ids[0]:  # смогли выписать роль на робота
            service_resource.attributes['staff_role_state'] = 'ok'
        if len(idm_role_ids) == 2:  # смогли выписать роль пользователя
            service_resource.attributes['staff_user_role_state'] = 'ok'
        service_resource.attributes['secret_role_state'] = 'ok'
        service_resource.save(update_fields=['attributes'])
        return self.get_external_id(service_resource), {}

    def edit(self, service_resource, prev_secret_id):
        vault_client = _get_vault_client()
        for param in (
            {'abc_scope': settings.ABC_ROBOTS_MANAGEMENT_SCOPE},
            {'abc_role_id': settings.ABC_ROBOTS_USERS_ROLE_ID},
        ):
            if not vault_client.add_user_role_to_secret(
                secret_uuid=service_resource.resource.attributes['secret_id']['value'],
                role=READER_ROLE,
                abc_id=service_resource.service.id,
                **param,
            ):
                raise SupplierError(f'Didn\'t grant access to robot password for {param}')

        for param in (
            {'abc_scope': settings.ABC_ROBOTS_MANAGEMENT_SCOPE},
            {'abc_role_id': settings.ABC_ROBOTS_USERS_ROLE_ID},
        ):
            if not vault_client.delete_user_role_from_secret(
                secret_uuid=prev_secret_id,
                role=READER_ROLE,
                abc_id=service_resource.service.id,
                **param
            ):
                raise SupplierError(f'Didn\'t remove access to robot password for {param}')

    def delete(self, service_resource, request):
        logger.info(f'Отзыв ресурса-робота {service_resource}')
        super(RobotsPlugin, self).delete(service_resource, request)
        if 'secret_id' not in service_resource.resource.attributes:
            return
        vault_client = _get_vault_client()
        secret_uuid = service_resource.resource.attributes['secret_id']['value']
        secret_dict = vault_client.get_secret(secret_uuid)
        secret_roles = secret_dict['secret_roles']
        for role in secret_roles:
            if 'abc_id' not in role:
                continue

            if role['abc_id'] == service_resource.service_id and role['role_slug'] == READER_ROLE:
                params = {}
                if role.get('abc_scope') == settings.ABC_ROBOTS_MANAGEMENT_SCOPE:
                    params['abc_scope'] = settings.ABC_ROBOTS_MANAGEMENT_SCOPE
                elif role.get('abc_role') == settings.ABC_ROBOTS_USERS_ROLE_ID:
                    params['abc_role_id'] = settings.ABC_ROBOTS_USERS_ROLE_ID
                if params:
                    # удаляем доступ
                    if not vault_client.delete_user_role_from_secret(
                        secret_uuid=secret_uuid,
                        role=READER_ROLE,
                        abc_id=service_resource.service_id,
                        **params
                    ):
                        raise SupplierError('Didn\'t remove access to robot password')

        return

    def reset_password(self, service_resource, request):
        resource = service_resource.resource

        if yenv.type != 'production':
            return resource.attributes

        with Session(oauth_token=settings.OAUTH_ROBOT_TOKEN) as session:
            response = session.post(
                settings.RESET_ROBOT_PASSWORD_URL,
                json={
                    'login': resource.name,
                    'secretId': resource.attributes['secret_id']['value'],
                }
            )
            if not response.ok:
                resource.attributes['error'] = force_text(response.content)
                resource.save(update_fields=['attributes'])
                raise SupplierError('Wrong answer during resetting password: %s', response.content)
            else:
                process_hd_api_response(response, resource)

            return resource.attributes

    def add_role_in_secret(self, resource, role, from_, to_):
        """
        Добавляет доступ role для _to в секрет указаннного робота от имени _from
        role: может быть OWNER или READER
        """

        message = f'Didn\'t grant access {role} to robot password for {to_}'
        if 'secret_id' not in resource.attributes:
            raise SupplierError(message)

        if role not in ['OWNER', 'READER']:
            raise SupplierError(f'Unknown role {role}')

        vault_client = _get_vault_client(login=from_)
        secret_uuid = resource.attributes['secret_id']['value']

        if not vault_client.add_user_role_to_secret(
            secret_uuid=secret_uuid,
            role=role,
            login=to_
        ):
            raise SupplierError(message)


@app.task
def add_role_robot_manager_to_staff(service_id, staff_id, robot_login):
    service = Service.objects.get(id=service_id)
    staff = Staff.objects.get(id=staff_id)
    staff_members = service.members.filter(staff=staff)
    role = Role.objects.globalwide().get(code=settings.ABC_ROBOTS_MANAGER_ROLE)
    logger.info(
        'begin to request robot manager role on creating robot. service:%s user:%s robot:%s',
        service.slug,
        staff.login,
        robot_login,
    )
    if staff_members.exists():
        if staff_members.responsibles().exists():
            user_status = 'heads'
            comment_ru = (
                'Роль выдана автоматически, так как сотрудник создал робота "%s" в сервисе "%s" '
                'и является одним из его руководителей'
            ) % (robot_login, service.name)
            comment_en = (
                'The role was granted because user created robot "%s" in service "%s" '
                'and the user is one of the managers of the service'
            ) % (robot_login, service.name_en)
        else:
            user_status = 'regular_member'
            comment_ru = (
                'Роль была запрошена, так как сотрудник создал робота "%s" в сервисе "%s". '
                'Если в команде сервиса нет ни одного участника с ролью Упраляющий роботами, робот не будет выдан.'
            ) % (robot_login, service.name)
            comment_en = 'The role was requested because user created robot "%s" in service "%s".'
            comment_en = comment_en % (robot_login, service.name_en)
    else:
        user_status = 'not_in_service'
        comment_ru = (
            'Роль была запрошена, так как сотрудник создал робота "%s" в сервисе "%s". '
            'На момент запроса сотрудник не состоит в данном сервисе, '
            'однако после выдачи роли сотрудник станет частью команды. '
            'Если в команде сервиса нет ни одного участника с ролью Упраляющий роботами, робот не будет выдан.'
        ) % (robot_login, service.name)
        comment_en = (
            'The role was requested because user created robot "%s" in service "%s". '
            'User is not a member of service for the moment of this role request '
            'but will be when the role granted.'
        ) % (robot_login, service.name_en)

    actions.request_membership(
        service,
        staff,
        role,
        comment='%s\n\n%s' % (comment_ru, comment_en),
        requester=staff,
    )
    logger.info(
        'Robot manager role requested on creating robot. service:%s user:%s robot:%s user_status:%s',
        service.slug,
        staff.login,
        robot_login,
        user_status,
    )
