import logging
from urllib.parse import urljoin

import requests
from django.conf import settings

from plan.api import exceptions
from plan.common.utils.oauth import get_token_for_zombie
from plan.common.utils.tvm import get_tvm_ticket
from plan.resources.exceptions import SupplierError
from plan.resources.suppliers.base import SupplierPlugin

log = logging.getLogger(__name__)


class TVMPlugin(SupplierPlugin):
    specific_actions = ['meta_info', 'recreate_secret', 'restore_secret', 'delete_old_secret']

    @property
    def service_ticket(self):
        return get_tvm_ticket(destination=str(settings.TVM_API_TVM_ID))

    @staticmethod
    def _handle_request(method, url, **kwargs):
        try:
            response = requests.request(method, url, **kwargs)

        except Exception as e:
            message = 'Cannot send request to TVM: {}'.format(e)
            log.error(message)
            raise SupplierError(message)

        log.info('TVM answer %s (%s)', response.content, response.status_code)

        if response.status_code == requests.codes.not_found:
            message = 'Wrong tvm url: {}'.format(url)
            log.error(message)
            raise SupplierError(message)

        if not response.ok:
            message = 'Invalid response code %s from tvm url %s' % (response.status_code, url)
            log.error(message)
            raise SupplierError(message)

        data = response.json()
        if data['status'] == 'error':
            if 'abc_team.member_required' in data['errors']:
                raise exceptions.PermissionDenied(
                    message={
                        'ru': 'Ошибка взаимодействия с TVM: у вас недостаточно прав для этого действия',
                        'en': 'TVM error: you are not allowed to do it'
                    }
                )

            else:
                raise exceptions.IntegrationError(
                    message={
                        'ru': 'Ошибка взаимодействия с TVM: %s' % ', '.join(data['errors']),
                        'en': 'TVM error: %s' % ', '.join(data['errors'])
                    }
                )

        return response

    def create(self, service_resource):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#createclient
        """
        if service_resource.resource.external_id:
            return service_resource.resource.external_id, service_resource.resource.supplier_response
        url = urljoin(settings.TVM_API, 'client/create?consumer=abc')
        data = {
            'uid': service_resource.requester.uid,
            'name': service_resource.resource.name,
            'abc_service_id': service_resource.service_id,
            'abc_request_id': service_resource.request_id
        }
        log.info('Send new resource %s to TVM (%s): %r', service_resource, settings.TVM_API, data)

        response = self._handle_request(
            method='post',
            url=url,
            data=data,
            verify=False,
            headers={'Authorization': f'OAuth {get_token_for_zombie()}'},
        )

        json = response.json()
        external_id = json.get('client_id')

        if not external_id:
            raise SupplierError('TVM did not return an external_id for resource!')

        return external_id, json

    def move_resource(self, service_resource, tvm_user_ticket):
        """
        Переносит tvm приложение на стороне tvm в указанный abc сервис
        """
        if not service_resource.resource.external_id:
            raise ValueError('Wrong TVM resource {}, no external_id'.format(service_resource.resource))
        if not tvm_user_ticket:
            raise ValueError('No user ticket supplied, {}'.format(service_resource.id))

        url = urljoin(settings.TVM_API, 'client/change_abc_service?consumer=abc')

        data = {
            'initiator_uid': service_resource.supplier_approver.uid,
            'abc_service_id': service_resource.service_id,
            'client_id': service_resource.resource.external_id
        }
        log.info('Send new resource %s to TVM (%s): %r', service_resource, settings.TVM_API, data)

        response = self._handle_request(
            method='post',
            url=url,
            data=data,
            verify=False,
            headers={
                'X-Ya-User-Ticket': tvm_user_ticket,
                'X-Ya-Service-Ticket': self.service_ticket,
            },
        )

        return response.json()

    def edit(self, service_resource, sender):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#editclient
        """
        if not service_resource.resource.external_id:
            raise ValueError('Wrong TVM resource {}, no external_id'.format(service_resource.resource))

        url = urljoin(settings.TVM_API, 'client/edit?consumer=abc')
        data = {
            'uid': sender.uid,
            'name': service_resource.resource.name,
            'client_id': service_resource.resource.external_id,
        }
        log.info('Edit resource %s in TVM (%s): %r', service_resource.resource, settings.TVM_API, data)

        self._handle_request(
            method='post',
            url=url,
            data=data,
            verify=False,
            headers={'Authorization': f'OAuth {get_token_for_zombie()}'},
        )

    def delete(self, service_resource, request):
        assert service_resource.resource.external_id, \
            'Wrong TVM resource %s, no external_id' % service_resource.resource

        data = {
            'uid': request.user.staff.uid,
            'client_id': service_resource.resource.external_id,
        }

        tvm_user_ticket = request.tvm_user_ticket
        if not tvm_user_ticket:
            log.warning('Marking request in tvm api with session_id')
            host = settings.TVM_API
            cookies = request.COOKIES.get('Session_id')
            headers = {
                'Ya-Client-Cookie': 'Session_id=%s; sessionid2=%s' %
                                    (cookies, request.COOKIES.get('sessionid2')),
                'Ya-Client-Host': 'abc.yandex-team.ru',
            }
        else:
            host = settings.TVM_API_V2
            headers = {
                'X-Ya-User-Ticket': tvm_user_ticket,
                'X-Ya-Service-Ticket': self.service_ticket,
            }

        url = urljoin(host, 'client/delete?consumer=abc')
        log.info('Delete resource %s in TVM (%s): %r', service_resource.resource, host, data)

        self._handle_request(
            method='post',
            url=url,
            data=data,
            verify=False,
            headers=headers,
        )

    def meta_info(self, service_resource, request):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#clientinfo
        """
        return self._secret_processing(service_resource, request, 'view')

    def recreate_secret(self, service_resource, request):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#recreateclientsecret
        """
        return self._secret_processing(service_resource, request, 'recreate')

    def restore_secret(self, service_resource, request):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#restoreoldclientsecret
        """
        return self._secret_processing(service_resource, request, 'restore')

    def delete_old_secret(self, service_resource, request):
        """
            https://wiki.yandex-team.ru/oauth/tvmabcapi/#deleteoldclientsecret
        """
        return self._secret_processing(service_resource, request, 'delete_old')

    def _secret_processing(self, service_resource, request, mode='view'):
        assert service_resource.resource.external_id, \
            'Wrong TVM resource %s, no external_id' % service_resource.resource

        url_action = {
            'view': ('info', 'GET'),
            'recreate': ('secret/recreate', 'POST'),
            'restore': ('secret/restore_old', 'POST'),
            'delete_old': ('secret/delete_old', 'POST'),
        }[mode]

        data = {
            'client_id': service_resource.resource.external_id,
        }

        tvm_user_ticket = request.tvm_user_ticket
        if not tvm_user_ticket:
            log.warning('Marking request in tvm api with session_id')
            host = settings.TVM_API
            cookies = request.COOKIES.get('Session_id')
            headers = {
                'Ya-Client-Cookie': 'Session_id=%s; sessionid2=%s' %
                                    (cookies, request.COOKIES.get('sessionid2')),
                'Ya-Client-Host': 'abc.yandex-team.ru',
            }
        else:
            host = settings.TVM_API_V2
            headers = {
                'X-Ya-User-Ticket': tvm_user_ticket,
                'X-Ya-Service-Ticket': self.service_ticket,
            }
        url = urljoin(host, 'client/{}?consumer=abc'.format(url_action[0]))

        log.info('Get metainfo for resource %s in TVM (%s): %r', service_resource.resource, host, data)

        try:
            response = requests.request(
                url_action[1],
                url=url,
                params=data,
                data=data,
                verify=False,
                headers=headers,
            )

            log.debug('Got metainfo for resource %s in TVM (%s): %s', service_resource.resource, url, response.content)

        except Exception as e:
            log.error('Cannot send request to TVM: %s', e)
            raise exceptions.IntegrationError(
                message={
                    'ru': 'Отправка запроса в TVM не удалась',
                    'en': 'Cannot send request to TVM',
                }
            )

        if response.status_code >= 300:
            log.error(
                'Unexpected status code %s while getting metainfo for %s from %s',
                response.status_code, service_resource, url
            )

            raise exceptions.IntegrationError(
                message={
                    'ru': 'Некорректный ответ от TVM',
                    'en': 'Incorrect TVM answer',
                }
            )

        data = response.json()

        if data['status'] == 'ok':
            attributes = data['content'].get('attributes', {}) \
                if isinstance(data['content'], dict) \
                else data['content'][0].get('attributes', {})

            return attributes

        elif data['status'] == 'error':
            if 'abc_team.member_required' in data['errors']:
                raise exceptions.PermissionDenied(
                    message={
                        'ru': 'Ошибка взаимодействия с TVM: у вас недостаточно прав для этого действия',
                        'en': 'TVM error: you are not allowed to do it'
                    }
                )

            elif 'old_secret.exists' in data['errors']:
                raise exceptions.Conflict(
                    message={
                        'ru': 'У приложения уже есть два секрета: для создания нового нужно предварительно удалить старый',
                        'en': 'The application already has two secrets: you must delete old secret to create a new one'
                    }
                )

            raise exceptions.IntegrationError(
                message={
                    'ru': 'Ошибка получения данных от TVM: %s' % ', '.join(data['errors']),
                    'en': 'TVM answer error: %s' % ', '.join(data['errors'])
                }
            )
