import csv
import itertools
import json
import hashlib
import logging
import operator
import datetime
import collections
from typing import Any, Dict

import time
import yenv
from tempfile import TemporaryFile

import six
from django.conf import settings
from django.contrib.auth.models import User
from django.core.urlresolvers import resolve
from django.db import transaction
from django.db.models import Count, Exists, OuterRef, Q
from django.db.utils import OperationalError, InterfaceError
from django.test.client import RequestFactory
from django.utils import timezone
from library.python.vault_client.errors import ClientError
from yt.yson import yson_to_json
import waffle

from plan.celery_app import app
from plan.common.person import Person
from plan.staff.models import Staff
from plan.common.utils.http import Session
from plan.common.utils.locks import locked_context
from plan.common.utils.oauth import get_abc_zombik
from plan.common.utils.tasks import (
    lock_task,
    task_metric,
    retry_func,
    get_last_success_start,
)
from plan.idm.adapters import RoleManager
from plan.idm.manager import idm_manager
from plan.notify.core.email import send_mail_basic
from plan.resources import constants
from plan.resources.exceptions import RobotIsDismissed, RobotsNotInTestingError, SupplierError, SupplierDelayingError
from plan.resources.importers import base, bot, conductor, crt, dispenser, femida, nanny, racktables, qloud
from plan.resources.importers.errors import ResourceImporterError
from plan.resources.models import (
    ResourceType,
    Resource,
    ServiceResource,
    ServiceResourceCounter,
    ResourceTag,
)
from plan.resources.suppliers.base import SupplierPlugin
from plan.resources.suppliers.robots import process_hd_api_response, RobotsPlugin
from plan.services.models import Service, ServiceTag
from plan.unistat.models import MetricsRecord
from plan.common.utils.dates import datetime_to_str, datetime_from_str

log = logging.getLogger(__name__)


def get_hash(*args) -> str:
    hash_params = ''.join(map(str, args))
    return hashlib.sha256(hash_params.encode()).hexdigest()


@retry_func(exceptions=(OperationalError, InterfaceError))
def grant_one_service_resource(service, external_keys_set, data, external_id, resource_params, check_tags=False):
    resource, _ = Resource.objects.get_or_create(**resource_params)
    data.update_db_resource(resource)
    if (external_id, data.type_name) not in external_keys_set:
        return
    log.info('Grant resource %s to service %s from supplier', resource, service)
    service_resource = ServiceResource.objects.filter(
        resource=resource,
        service=service,
    ).first()
    if service_resource:
        service_resource.forced_grant(check_tags)
        service_resource.save()
    else:
        ServiceResource.objects.create_granted(
            resource=resource,
            service=service,
            type_id=resource.type_id,
        )


@transaction.atomic
def grant_chunk_service_resources(service, external_keys_set, types, external_data):
    for data in external_data:
        external_id = six.text_type(data.external_id)
        resource_params = {
            'type': types[data.type_name],
            'external_id': external_id,
        }
        grant_one_service_resource(
            service,
            external_keys_set,
            data,
            external_id,
            resource_params,
            check_tags=True,
        )


@transaction.atomic
def grant_chunk_service_resources_in_bulk(service, external_keys_set, types, external_data):
    resources_map = {}
    for type_name, type_data in itertools.groupby(external_data, key=operator.attrgetter('type_name')):
        data_map = {str(data.external_id): data for data in type_data}

        type_resources = Resource.objects.filter(type=types[type_name], external_id__in=data_map.keys())
        for resource in type_resources:
            data = data_map.pop(resource.external_id)
            data.update_db_resource(resource)

        creating = []
        for external_id, data in data_map.items():
            creating.append(
                Resource(
                    type=types[type_name],
                    external_id=external_id,
                    name=data_map[external_id].name,
                    attributes=data.attributes,
                    link=data.link,
                )
            )
        created = Resource.objects.bulk_create(creating, ctt_considered=True)
        for resource in created:
            resource._closure_createlink()

        for resource in itertools.chain(type_resources, created):
            if (resource.external_id, type_name) in external_keys_set:
                resources_map[resource.id] = resource

    resource_id_set = set(resources_map.keys())
    existing = ServiceResource.objects.filter(service=service, resource_id__in=resource_id_set)
    existing.exclude(state=ServiceResource.GRANTED).forced_grant()

    prepared_relations = []
    granted_at = timezone.now()
    granter = get_abc_zombik()
    for resource_id in (resource_id_set - set(existing.values_list('resource_id', flat=True))):
        resource = resources_map[resource_id]
        prepared_relations.append(
            ServiceResource(
                resource=resource,
                service=service,
                type_id=resource.type_id,
                state=ServiceResource.GRANTED,
                granter=granter,
                granted_at=granted_at,
            )
        )
    ServiceResource.objects.bulk_create(prepared_relations)


def grant_service_resources(service, external_keys_set, types, external_data, chunk_size=1000, check_tags=False):
    external_data = sorted(external_data, key=operator.attrgetter('type_name'))
    for i in range(0, len(external_data), chunk_size):
        chunk = external_data[i:i + chunk_size]
        if check_tags:
            grant_chunk_service_resources(service, external_keys_set, types, chunk)
        else:
            grant_chunk_service_resources_in_bulk(service, external_keys_set, types, chunk)


def deprive_service_resources(service, external_keys_set, use_code=False):
    """Отозвать ресурсы и проверить необходимость снять теги сервиса."""
    for external_id, type_name_or_code in external_keys_set:
        query = {
            'resource__external_id': external_id,
            'service': service,
            'state__in': ServiceResource.ALIVE_STATES,
        }
        if use_code:
            query['type__code'] = type_name_or_code
        else:
            query['type__name'] = type_name_or_code
        service_resource = ServiceResource.objects.get(**query)
        log.info('Deprive resource %s of service %s from supplier', external_id, service)
        service_resource.forced_deprive(check_tags=True)
        service_resource.save()


def deprive_service_resources_in_bulk(service, external_keys_set, use_code=False, chunk_size=1000):
    """Отозвать ресурсы пакетно."""
    resources_per_type = {}
    ids_per_type = {}
    for external_id, type_name_or_code in external_keys_set:
        resources_per_type.setdefault(
            type_name_or_code,
            _make_service_resource_query(service, type_name_or_code, use_code)
        )
        ids_per_type.setdefault(type_name_or_code, []).append(external_id)
    for type_name_or_code, query in resources_per_type.items():
        type_resource_ids = ids_per_type[type_name_or_code]
        for i in range(0, len(type_resource_ids), chunk_size):
            chunk = type_resource_ids[i:chunk_size]
            depriving_resources = ServiceResource.objects.filter(
                resource__external_id__in=chunk, **query)
            log.info(
                'Deprive resources of type %s for service %s: %s',
                type_name_or_code, service, chunk,
            )
            depriving_resources.forced_deprive()


def _make_service_resource_query(service, type_name_or_code, use_code):
    query = {
        'service': service,
        'state__in': ServiceResource.ALIVE_STATES,
    }
    if use_code:
        query['type__code'] = type_name_or_code
    else:
        query['type__name'] = type_name_or_code
    return query


def sync_service_resource(
    supplier, service, types, external_data, use_code=False,
    start_time=None, check_tags=False,
    resource_type=None
):
    """Хочется переехать с name на code везде, но пока есть параметр use_code для совместимости.
    Нужно стремиться, чтобы он стал True везде"""
    keys_from_source = {(six.text_type(data.external_id), data.type_name) for data in external_data}
    resources_qs = (
        Resource.objects.filter(
            serviceresource__service=service,
            serviceresource__state__in=ServiceResource.ALIVE_STATES,
            type__supplier=supplier,
        )
    )
    if resource_type is not None:
        resources_qs = resources_qs.filter(type=resource_type)
    if start_time is not None:
        resources_qs = (
            resources_qs.filter(
                created_at__lt=start_time - constants.SERVICE_SYNC_TIME_LIMIT,
            )
        )
    if use_code:
        keys_resource = set(resources_qs.values_list('external_id', 'type__code'))
    else:
        keys_resource = set(resources_qs.values_list('external_id', 'type__name'))

    keys_to_grant = keys_from_source - keys_resource
    keys_to_deprive = keys_resource - keys_from_source
    grant_service_resources(service, keys_to_grant, types, external_data, check_tags=check_tags)
    if check_tags:
        deprive_service_resources(service, keys_to_deprive, use_code=use_code)
    else:
        deprive_service_resources_in_bulk(service, keys_to_deprive, use_code=use_code)


def create_or_update_bot_types(supplier, existing_types, external_types):
    existing = ResourceType.objects.filter(supplier=supplier, name__in=external_types)
    existing_types.update({rt.name: rt for rt in existing})
    missing = external_types - set(existing_types.keys())
    for type_name in missing:
        existing_types[type_name] = ResourceType.objects.create(
            supplier=supplier,
            name=type_name,
            is_important=True,
        )
    return existing_types


@lock_task(
    last_finished_metric_name='sync_with_bot',
    lock_key=lambda service_slug='', *args, **kwargs: 'sync_with_bot' + service_slug
)
def sync_with_bot(start_time=None, service_slug=None):
    if start_time:
        start_time = datetime_from_str(start_time)
    supplier = Service.objects.get(slug=constants.BOT_SUPPLIER_SERVICE_SLUG)
    bad_service_slugs = settings.EXCLUDE_IN_SYNC_WITH_BOT
    if service_slug:
        services = Service.objects.filter(slug=service_slug)
    elif bad_service_slugs:
        services = Service.objects.exclude(slug__in=bad_service_slugs)
    else:
        services = Service.objects.all()

    existing_types = {}
    for service in services:
        try:
            external_data = bot.get_associated_data(service.id)
        except ResourceImporterError as error:
            # NOTE(lavrukov): Для неактивных проектов bot обычно возвращает 404
            if service.state in Service.states.ACTIVE_STATES:
                log.warning(
                    'Can\'t get bot info for service %s: %s',
                    service, getattr(error, 'message', str(error))
                )
            continue

        existing_types = create_or_update_bot_types(supplier, existing_types, external_data['type_names'])
        sync_service_resource(supplier, service, existing_types, external_data['servers'], start_time=start_time)


@lock_task(last_finished_metric_name='sync_with_oebs')
def sync_with_oebs(service_id=None):
    hr_resource_type = ResourceType.objects.get(code=constants.HR_RESOURCETYPE_CODE)
    types = {
        constants.HR_RESOURCETYPE_CODE: hr_resource_type
    }

    qs = Service.objects.all()
    if service_id:
        qs = qs.filter(pk=service_id)

    for service in qs:
        try:
            external_data = bot.get_associated_data(service.id)['service']

        except ResourceImporterError as error:
            # NOTE: Для неактивных проектов bot обычно возвращает 404
            if service.state in Service.states.ACTIVE_STATES:
                log.warning('Cannot get oebs info for service %s: %s', service, getattr(error, 'message', str(error)))

            if error.status_code == 404:
                log.warning('Got 404 for service %s, depriving the hr resources', service.slug)
                ServiceResource.objects.filter(type=hr_resource_type, service=service).forced_deprive()
            continue

        drop_hr_resource = False
        reason = None
        if external_data:
            if external_data.is_hr:
                external_data['service_id'] = service.id
                sync_service_resource(
                    hr_resource_type.supplier,
                    service,
                    types,
                    [external_data],
                    use_code=True,
                    check_tags=True,
                    resource_type=hr_resource_type,
                )
            else:
                drop_hr_resource = True
                reason = 'Bot told us the service is not hr anymore'
        else:
            drop_hr_resource = True
            reason = 'Bot returned nothing about service'

        if drop_hr_resource:
            service_resources = ServiceResource.objects.filter(type=hr_resource_type, service=service)
            count = service_resources.count()
            if count:
                log.warning('We are going to deprive oebs resources for services %s. Reason: %s', service.pk, reason)
                service_resources.forced_deprive()
                log.warning('We have deprived %s oebs resources for service %s. Reason: %s', count, service.pk, reason)


@lock_task(last_finished_metric_name='sync_with_racktables')
def sync_with_racktables():
    try:
        external_data = racktables.get_data()
    except ResourceImporterError as error:
        log.exception('Can\'t get racktables info: %s', getattr(error, 'message', str(error)))
        raise

    supplier = Service.objects.get(slug=constants.RACKTABLES_SUPPLIER_SERVICE_SLUG)
    type_names = set()
    for data in external_data.values():
        type_names |= {resource.type_name for resource in data}
    types = create_or_update_bot_types(supplier, {}, type_names)
    services = {service.id: service for service in Service.objects.filter(id__in=external_data.keys())}

    for service_id, service_data in external_data.items():
        try:
            service = services[service_id]
        except KeyError:
            log.warning('Racktables return invalid service %s', service_id)
            continue

        sync_service_resource(supplier, service, types, service_data)

    resources_to_deprive = (
        ServiceResource.objects
        .alive()
        .filter(resource__type__in=types.values())
        .exclude(service_id__in=external_data.keys())
    )
    resources_to_deprive.forced_deprive()


def get_or_create_type(supplier, external_data_cls):
    type_name = external_data_cls.TYPE_NAME
    rtype, _ = ResourceType.objects.get_or_create(supplier=supplier, name=type_name)
    return rtype


def get_conductor_types(supplier):
    project_type = get_or_create_type(supplier, conductor.ConductorProject)
    group_type = get_or_create_type(supplier, conductor.ConductorGroup)
    host_type = get_or_create_type(supplier, conductor.ConductorHost)
    return {
        project_type.name: project_type,
        group_type.name: group_type,
        host_type.name: host_type,
    }


@lock_task(last_finished_metric_name='sync_with_conductor')
def sync_with_conductor(start_time=None):
    if start_time:
        start_time = datetime_from_str(start_time)
    supplier = Service.objects.get(slug=constants.CONDUCTOR_SUPPLIER_SERVICE_SLUG)
    types = get_conductor_types(supplier)
    services_projects = conductor.get_services_projects()

    for service_id, projects in services_projects.items():
        try:
            service = Service.objects.get(id=service_id)
        except Service.DoesNotExist:
            log.warning('Conductor return project with invalid ABC id %s', service_id)
            continue

        try:
            groups = conductor.get_projects_groups(projects)
            hosts = conductor.get_projects_hosts(projects)
        except ResourceImporterError as error:
            project_names = [project.name for project in projects]
            log.exception('Can\'t fetch info from conductor for projects %s: %s', project_names, error)
            continue

        external_data = list(itertools.chain(projects, groups, hosts))
        sync_service_resource(supplier, service, types, external_data, start_time=start_time)

    # gone resources
    conductor_services = Service.objects.filter(
        serviceresource__type__supplier_id=supplier.id,
        serviceresource__state__in=ServiceResource.ACTIVE_STATES,
    )
    for service in conductor_services:
        if service.id not in services_projects:
            sync_service_resource(supplier, service, types, [], start_time=start_time)


@lock_task()
def sync_resources(resource_type_id=None):
    start_time = datetime_to_str(timezone.now())
    types = ResourceType.objects.filter(import_plugin__in=list(base.Plugin.get_synced()))
    if resource_type_id:
        types = types.filter(pk=resource_type_id)
    else:
        types = types.filter(is_enabled=True)

    for resource_type in types:
        sync_resource_type.apply_async(kwargs={
            'resource_type_id': resource_type.pk,
            'start_time': start_time
        })


@lock_task(lock=False)
def sync_resource_type(resource_type_id, start_time=None):
    if start_time:
        start_time = datetime_from_str(start_time)
    resource_type = ResourceType.objects.get(pk=resource_type_id)
    lock_name = 'sync_resource_{}'.format(resource_type.code)
    with locked_context(lock_name, block=False) as acquired:
        if not acquired:
            log.info('Lock for resource sync of type %s already taken', resource_type.code)
            return
        log.info('Import data for %s started', resource_type)

        try:
            with task_metric('sync_resource_{}'.format(resource_type.code)):
                plugin_class = base.Plugin.get_plugin_class(resource_type.import_plugin)
                plugin = plugin_class(resource_type=resource_type)

                data = plugin.fetch()

                services_with_type_pks = []

                for record in data:
                    sync_service_resources(
                        resource_type,
                        record['service'],
                        record['resources'],
                        start_time=start_time
                    )
                    services_with_type_pks.append(record['service'].pk)

                query = (
                    ServiceResource.objects
                    .filter(resource__type=resource_type)
                    .granted()
                    .exclude(service_id__in=services_with_type_pks)
                )
                if start_time:
                    query = query.filter(resource__created_at__lt=start_time - constants.SERVICE_SYNC_TIME_LIMIT)
                query.forced_deprive()

                log.info('Import data for %s finished', resource_type)
                plugin.sync_state()
                log.info('Finished syncing state for %s', resource_type)

        except Exception as e:
            log.exception('Cannot sync resources from %s: %s', resource_type, e)


def sync_service_resources(resource_type, service, service_data, start_time=None):
    query = (
        ServiceResource.objects
        .filter(
            service=service,
            state__in=ServiceResource.ACTIVE_STATES,
            resource__type=resource_type,
        )
        .select_related('resource')
    )
    if start_time is not None:
        query = (
            query.filter(resource__created_at__lt=start_time - constants.SERVICE_SYNC_TIME_LIMIT)
        )

    service_resources = {sr.resource.external_id: sr.id for sr in query}
    resources_from_source = refresh_resources(resource_type, service_data)
    update_relations(service, resource_type, resources_from_source, service_resources)


def update_relations(service, resource_type, resources_from_source, service_resources):
    from_source_ids = set(resources_from_source.keys())
    service_resource_ids = set(service_resources.keys())

    for resource_id in from_source_ids - service_resource_ids:
        log.info('Grant resource %s to service %s from supplier',
                 resources_from_source[resource_id], service)

        with transaction.atomic():
            resource = resources_from_source[resource_id]
            sr, _ = ServiceResource.objects.get_or_create(
                resource=resource,
                service=service,
                defaults={'type_id': resource.type_id},
            )
            sr.forced_grant()
            sr.save()

    service_resources_pks = [service_resources[resource_id] for resource_id in service_resource_ids - from_source_ids]
    log.info('Deprive service resources %s of service %s', service_resources_pks, service)
    ServiceResource.objects.filter(pk__in=service_resources_pks).forced_deprive()

    if len(service_resource_ids) == 0 and len(from_source_ids) > 0:
        tags = resource_type.service_tags.exclude(pk__in=service.tags.all())
        service.tags.add(*tags)


def refresh_resources(resource_type, service_data):
    resources = {}
    for data in service_data:
        resource_params = {
            'type': resource_type,
            'external_id': data['id'],
        }
        resource, created = Resource.objects.get_or_create(**resource_params)
        if created or resource_type.code in settings.FORCE_RESOURCE_ATTRIBUTES_UPDATE:
            update_db_resource(resource, data)
        resources[resource.external_id] = resource
    return resources


def update_db_resource(resource, data):
    """Обновляет и апдейтит ресурс, если это необходимо"""
    changed = False
    for attr in ['name', 'attributes', 'link']:
        external_field = data.get(attr)
        obj_field = getattr(resource, attr)
        if obj_field != external_field:
            setattr(resource, attr, external_field)
            changed = True

    if changed:
        resource.save()


@lock_task
def sync_certificates_incremental(last_start=None):
    if last_start is not None:
        last_start = datetime_from_str(last_start)
    else:
        last_start = get_last_success_start('sync_certificates_incremental')
        if last_start is not None:
            last_start -= datetime.timedelta(minutes=10)
    cert_resource_type = ResourceType.objects.get(code='cert')
    plugin = crt.CrtPlugin(resource_type=cert_resource_type)
    data = plugin.fetch_incremental(last_start)
    for record in data:
        service = record['service']
        granted, deprived = refresh_certificates_incremental(cert_resource_type, record['resources'])
        for resource in granted.values():
            log.info('Grant certificate %s to service %s', resource.external_id, service)
            existing = ServiceResource.objects.filter(resource=resource, type=cert_resource_type)
            sr = existing.filter(service=service).first()
            if sr is None:
                sr = ServiceResource(resource=resource, service=service, type=cert_resource_type)
            if sr.state != ServiceResource.GRANTED:
                sr.forced_grant()
            sr.save()
            if existing.exclude(id=sr.id).granted().exists():
                existing.exclude(id=sr.id).granted().forced_deprive()

        log.info('Deprive certificates %s', ', '.join(str(k) for k in deprived.keys()))
        ServiceResource.objects.filter(resource_id__in=deprived.keys()).forced_deprive()


def refresh_certificates_incremental(resource_type, service_data):
    granted = {}
    deprived = {}
    for data in service_data:
        resource_params = {
            'type': resource_type,
            'external_id': data['id'],
        }
        if data['attributes']['status'] == 'issued':
            resource, created = Resource.objects.get_or_create(**resource_params)
            if created:
                update_db_resource(resource, data)
            granted[resource.id] = resource
        else:
            resource = Resource.objects.filter(**resource_params).first()
            if resource is not None:
                deprived[resource.id] = resource
    return granted, deprived


@transaction.atomic
def sync_one_resource(obj, resource_type, active_resources_pks):
    resource, _ = Resource.objects.update_or_create(
        defaults=obj.not_required_params, type=resource_type, **obj.resource_params
    )
    active_resources_pks.append(resource.pk)
    if not obj.services.exists():
        ServiceResource.objects.filter(resource=resource).forced_deprive()
        return False
    else:
        used_service_resources_pks = []

        for service in obj.services:
            service_resource, created = ServiceResource.objects.get_or_create(
                resource=resource,
                service=service,
                defaults={'type_id': resource.type_id},
            )
            used_service_resources_pks.append(service_resource.pk)
            if created:
                service_resource.approve(get_abc_zombik())

            elif service_resource.state in ServiceResource.BAD_STATES:
                service_resource.state = ServiceResource.APPROVED
                service_resource.save(update_fields=['state', ])
                service_resource.grant(Person(get_abc_zombik()))

        ServiceResource.objects.filter(resource=resource).exclude(pk__in=used_service_resources_pks).forced_deprive()

        return True


def deprive_inactive_resources(resource_type, active_resources_pks):
    inactive_resources = Resource.objects.filter(type=resource_type).exclude(pk__in=active_resources_pks)
    ServiceResource.objects.filter(resource__in=inactive_resources, state=ServiceResource.GRANTED).forced_deprive()


@lock_task(last_finished_metric_name='sync_with_nanny')
def sync_with_nanny():
    services_with_abc = total_services = 0
    active_resources_pks = []
    resource_type = ResourceType.objects.get(code=settings.NANNY_RESOURCE_TYPE_CODE)
    for nanny_service in nanny.get_services(resource_type.import_link):
        total_services += 1
        services_with_abc += sync_one_resource(nanny_service, resource_type, active_resources_pks)

    deprive_inactive_resources(resource_type, active_resources_pks)
    metrcis_record, _ = MetricsRecord.objects.get_or_create()
    metrcis_record.metrics.update({
        'nanny_services_with_abc': services_with_abc,
        'nanny_services_without_abc': total_services - services_with_abc,
    })
    metrcis_record.save()


@lock_task(last_finished_metric_name='sync_with_qloud')
def sync_with_qloud():
    projects_with_abc = apps_with_abc = total_projects = total_apps = 0
    for project_resource_type_code, application_resource_type_code in settings.QLOUD_IMPORT_SETTINGS:
        active_projects_pks = []
        active_apps_pks = []
        project_resource_type = ResourceType.objects.get(code=project_resource_type_code)
        application_resource_type = ResourceType.objects.get(code=application_resource_type_code)
        for project in qloud.get_projects(project_resource_type.import_link):
            total_projects += 1
            projects_with_abc += sync_one_resource(project, project_resource_type, active_projects_pks)
            for application in qloud.get_applications(application_resource_type.import_link, project):
                total_apps += 1
                apps_with_abc += sync_one_resource(application, application_resource_type, active_apps_pks)

        deprive_inactive_resources(project_resource_type, active_projects_pks)
        deprive_inactive_resources(application_resource_type, active_apps_pks)

    metrics_record, _ = MetricsRecord.objects.get_or_create()
    metrics_record.metrics.update({
        'qloud_projects_with_abc': projects_with_abc,
        'qloud_projects_without_abc': total_projects - projects_with_abc,
        'qloud_apps_with_abc': apps_with_abc,
        'qloud_apps_without_abc': total_apps - apps_with_abc
    })
    metrics_record.save()


@lock_task(last_finished_metric_name='sync_with_dispenser')
def sync_with_dispenser():
    projects = list(dispenser.get_projects())

    for project in projects:
        if project.service is None:
            resource = project.get_or_create_resource()
            ServiceResource.objects.filter(resource=resource).forced_deprive()
        else:
            project.get_or_create_service_resource()

    # находим проекты, которые удалены из источника
    deleted_projects = (
        Resource
        .objects
        .filter(type__code='dispenser_project')
        .exclude(external_id__in=[project.key for project in projects])
    )

    ServiceResource.objects.filter(resource__in=deleted_projects).forced_deprive()


@lock_task(last_finished_metric_name='sync_with_femida')
def sync_with_femida():
    active_resources_pks = []
    resource_type = ResourceType.objects.get(code=settings.FEMIDA_RESOURCE_TYPE_CODE)

    for vacancy in femida.get_all_vacancies(resource_type.import_link):
        sync_one_resource(vacancy, resource_type, active_resources_pks)

    deprive_inactive_resources(resource_type, active_resources_pks)


@app.task
def grant_missing_idm_roles_for_resource_type(resource_type_id):
    resource_type = ResourceType.objects.get(pk=resource_type_id)
    manager = idm_manager()
    lock_name = f'grant_missing_idm_roles_for_resource_type_{resource_type.code}'

    with task_metric(f'grant_missing_idm_roles_for_{resource_type.code}', send_to_unistat=True), \
            locked_context(lock_name, block=False) as acquired:

        if not acquired:
            log.warning('Lock %s was already taken', lock_name)
            return

        idm_resources = ServiceResource.objects.filter(type_id=resource_type, state=ServiceResource.GRANTED)
        service_resource = idm_resources.first()
        if service_resource is None:
            log.warning('No serviceresources of type %s', resource_type.code)

        plugin = SupplierPlugin.get_plugin_class(resource_type.supplier_plugin)()
        roles_data = plugin.build_roles_data_for_idm_sync(service_resource)
        remote_roles = set()

        for role in roles_data:
            remote_roles.update(
                plugin.role_dict_to_key(role, from_api=True)
                for role in RoleManager.get_roles(manager, role, light=True)['objects']
            )

        for service_resource in idm_resources.iterator():
            # Для некоторых ресурсов нужно меньше ролей чем обычно. В таких случаях build_role_data возвращает список,
            # в котором есть None
            try:
                required_roles = filter(None, plugin.build_role_data(service_resource))
            except RobotIsDismissed:
                # робот уволен, делать ничего не нужно, ресурс уберет таска revoke_dismissed_robots_resources
                continue
            except SupplierDelayingError:
                # нет возможности выдать такую роль, так что сверять ее не нужно
                continue
            role_data = {plugin.role_dict_to_key(role, from_api=False) for role in required_roles}

            if not role_data.issubset(remote_roles):
                if waffle.switch_is_active('grant_missing_idm_roles'):
                    try:
                        api_responses = plugin.create_missing(service_resource)
                    except Exception:
                        log.exception('Could not actualize idm roles for service_resource=%s', service_resource.pk)
                    else:
                        if api_responses is None:
                            continue  # фактически ничего не поменялось
                        service_resource.resource.supplier_response = api_responses
                        service_resource.resource.save(update_fields=['supplier_response'])
                else:
                    log.info(
                        'ServiceResource %s of type %s is missing in idm', service_resource.id, resource_type.code
                    )
            else:
                log.info('ServiceResource %s of type %s exists in idm', service_resource.id, resource_type.code)


@lock_task(last_finished_metric_name='grant_missing_idm_roles')
def grant_missing_idm_roles():
    resource_types = ResourceType.objects.filter(code__in=settings.SYNCABLE_WITH_IDM_RESOURCES)

    for resource_type_id in resource_types.values_list('pk', flat=True):
        grant_missing_idm_roles_for_resource_type.apply_async(args=[resource_type_id])


@lock_task(last_finished_metric_name='revoke_obsolete_direct_roles')
def revoke_obsolete_direct_roles():
    last_exc = None
    resource_type = ResourceType.objects.get(code=settings.DIRECT_RESOURCE_TYPE_CODE)
    plugin = SupplierPlugin.get_plugin_class(resource_type.supplier_plugin)()

    obsolete_service_resources = ServiceResource.objects.filter(
        type=resource_type,
        state=ServiceResource.OBSOLETE,
        attributes__role_id__isnull=False,
    )
    for service_resource in obsolete_service_resources:
        try:
            plugin.revoke_only_personal_roles(service_resource)
        except Exception as e:
            last_exc = e

    if last_exc is not None:
        raise last_exc


@lock_task
def revoke_dismissed_robots_resources():
    modified_from = timezone.now() - timezone.timedelta(days=7)
    dismissed_robots = Staff.objects.filter(
        is_robot=True, is_dismissed=True,
        modified_at__gte=modified_from,
    ).values_list('login', flat=True)

    if dismissed_robots:
        robots_resources = ServiceResource.objects.filter(
            type__code='staff-robot',
            state__in=ServiceResource.ALIVE_STATES,
            resource__name__in=dismissed_robots,
        )
        plugin = RobotsPlugin()
        for service_resource in robots_resources:
            try:
                plugin.delete(service_resource, None)
                service_resource.state = ServiceResource.DEPRIVED
                service_resource.attributes.pop('role_id', None)
                service_resource.save(update_fields=['attributes', 'state'])
            except (ClientError, SupplierDelayingError):
                log.error('Didn\'t revoke service_resource %s', service_resource)


@lock_task(last_finished_metric_name='deprive_financial_resource_connected_with_inactive_services')
def deprive_financial_resource_connected_with_inactive_services():
    last_exc = None
    resource_types = ResourceType.objects.filter(
        code__in=(settings.DIRECT_RESOURCE_TYPE_CODE, settings.METRIKA_RESOURCE_TYPE_CODE)
    )
    for resource_type in resource_types:
        plugin = SupplierPlugin.get_plugin_class(resource_type.supplier_plugin)()
        service_resources_of_inactive_services = (
            ServiceResource.objects
            .filter(
                type=resource_type,
            )
            .exclude(
                service__state__in=Service.states.ACTIVE_STATES,
            )
        )

        service_resources_to_deprive = service_resources_of_inactive_services.filter(state=ServiceResource.GRANTED)
        for service_resource in service_resources_to_deprive:
            try:
                plugin.delete(service_resource, None)
                service_resource.state = 'deprived'
                service_resource.attributes.pop('role_id', None)
                service_resource.save(update_fields=['attributes', 'state'])
            except Exception:
                log.exception('Error during depriving service_resource %s', service_resource.id)

        service_resources_to_force_deprive = service_resources_of_inactive_services.filter(
            state__in=(ServiceResource.REQUESTED, ServiceResource.APPROVED)
        )
        service_resources_to_force_deprive.update(state=ServiceResource.DEPRIVED)

    if last_exc is not None:
        raise last_exc


@app.task
def grant_resource(service_resource_id):
    try:
        service_resource = ServiceResource.objects.get(id=service_resource_id)
    except ServiceResource.DoesNotExist:
        log.info('Service resource with id <%s> not found', service_resource_id)
        return

    log.info('Attempt to grant %s', service_resource)
    if service_resource.state != ServiceResource.GRANTING:
        log.info('Cannot grant %s - wrong state %s', service_resource, service_resource.state)
        return

    try:
        if service_resource.type.supplier_plugin:
            _send_resource_to_supplier(service_resource)

        if service_resource.obsolete:
            service_resource.obsolete.replace()
            service_resource.obsolete.save()
            log.info('%s become obsolete', service_resource.obsolete)

        service_resource._grant()
        service_resource.save()
        log.info('%s granted', service_resource)
    except (SupplierDelayingError, ClientError) as e:
        log.warning(
            '%s. The robot has not moved to the status GRANTED: %s', service_resource, getattr(e, 'message', str(e))
        )

    except Exception:
        log.exception('Cannot grant resource %s', service_resource)
        service_resource.grant_retries -= 1
        update_fields = ['grant_retries']
        if service_resource.grant_retries <= 0:
            service_resource.fail()
            update_fields.append('state')

        service_resource.save(update_fields=update_fields)


def _send_resource_to_supplier(service_resource):
    plugin = SupplierPlugin.get_plugin_class(service_resource.type.supplier_plugin)()

    log.info('Trying to request %s', service_resource)

    external_id, api_response = plugin.create(service_resource)

    if service_resource.resource.external_id != external_id:
        service_resource.resource.external_id = external_id
        service_resource.resource.supplier_response = api_response
        service_resource.resource.save()


@lock_task
def retry_grant():
    threshold = timezone.now() - timezone.timedelta(seconds=settings.ABC_RETRY_GRANT_TRESHOLD)
    qs = ServiceResource.objects.filter(
        state=ServiceResource.GRANTING, modified_at__lte=threshold
    ).values_list('id', flat=True)

    for service_resource_id in qs:
        log.info('Try to regrant ServiceResource %s', service_resource_id)
        grant_resource.apply_async(args=[service_resource_id])


def _render_service_resource(service_resource):
    context = {}
    resource = service_resource['resource']

    attributes_list = [
        'ID:%s' % resource.get('external_id'),
        'name:%s' % resource.get('name')
    ]
    for attr in resource.get('attributes', []):
        attributes_list.append('%s:%s' % (attr['name'], attr.get('value')))

    context['supplier'] = resource['type']['supplier']['name']['en']
    context['type'] = resource['type']['name']['en']
    context['consumer'] = service_resource['service']['name']['en']
    context['consumer_id'] = service_resource['service']['id']
    context['consumer_slug'] = service_resource['service']['slug']
    context['tags'] = ';'.join([x['slug'] for x in (service_resource['tags'] + service_resource['supplier_tags'])])
    context['attributes'] = ';'.join(attributes_list)
    context['modified_at'] = service_resource['modified_at']
    context['state'] = service_resource['state']
    context['has_monitoring'] = service_resource['has_monitoring']
    context['need_monitoring'] = resource['type']['need_monitoring']

    return context


def _get_json_response_from_view(path, requester, params):
    request = RequestFactory().get(path, data=params)
    request.user = requester
    view_func = resolve(path).func
    response = view_func(request)
    response.render()
    return json.loads(response.content)


@app.task
def send_csv_from_view(path, requester_username, params=None):
    params = params or {}
    params['page'] = 1
    params['page_size'] = constants.CSV_VIEW_PAGE_SIZE
    params['no_actions'] = True
    requester = User.objects.get(username=requester_username)
    response = _get_json_response_from_view(path, requester, params)
    with TemporaryFile('r+t') as temp_file:
        csv_writer = csv.DictWriter(
            temp_file,
            fieldnames=[
                'supplier', 'type', 'consumer', 'consumer_id', 'consumer_slug',
                'tags', 'attributes', 'modified_at', 'state',
                'has_monitoring', 'need_monitoring',
            ],
            quoting=csv.QUOTE_ALL,
            lineterminator='\n',
        )
        csv_writer.writeheader()
        while True:
            for service_resource in response['results']:
                csv_writer.writerow(_render_service_resource(service_resource))
            if response.get('next'):
                params['page'] += 1
                response = _get_json_response_from_view(path, requester, params)
            else:
                break

        temp_file.seek(0)
        send_mail_basic(
            subject='resources',
            body='',
            to=[requester.email],
            attach=('resources.csv', temp_file.read())
        )


@lock_task
def poll_failed_robot_passwords():
    with Session(oauth_token=settings.OAUTH_ROBOT_TOKEN) as session:
        for resource in Resource.objects.filter(type__code='staff-robot').exclude(attributes__pass_status='ok'):
            response = session.get(
                settings.CHECK_ROBOT_PASSWORD_STATUS_URL.format(operation=resource.attributes['operation'])
            )
            if response.ok:
                process_hd_api_response(response, resource)


def actualize_service_tag(service_tag):
    pks = ServiceResource.objects.granted().filter(type__in=service_tag.resource_types.all()).values('service')
    for service in Service.objects.filter(Q(pk__in=pks) & ~Q(tags=service_tag)):
        service.tags.add(service_tag)
    for service in Service.objects.filter(~Q(pk__in=pks) & Q(tags=service_tag)):
        service.tags.remove(service_tag)


@lock_task
def actualize_service_tags():
    for tag in ServiceTag.objects.prefetch_related('resource_types').filter(resource_types__isnull=False):
        actualize_service_tag(tag)


@lock_task
def update_serviceresources_count(days_for=None, service_id=None):
    """
    Обновляем счетчики по всем типам ресурсов

    Берем все ресурсы измененные с последнего запуска (или
    переданного значения) и для каждого типа и всех сервисов, у которых
    ресурсы этого типа были изменены - пересчитываем счетчики
    """

    log.info('Updating ServiceResourceCounter')
    if days_for:
        since_dt = timezone.now() - datetime.timedelta(days=days_for)
    else:
        since_dt = get_last_success_start('update_serviceresources_count')

    # соберем все связки service/resource_type, где могло произойти обновление количества
    query = {'modified_at__gte': since_dt}
    if service_id:
        query['service_id'] = service_id
    updated_resources = ServiceResource.objects.filter(**query).values_list(
        'service_id', 'type_id',
    ).order_by().distinct()

    types_by_services = collections.defaultdict(list)

    for service_id, type_id in updated_resources:
        types_by_services[type_id].append(service_id)

    to_create = []
    to_delete = []

    for type_id, services_ids in types_by_services.items():
        _update_serviceresources_count_by_type(
            type_id=type_id,
            services_ids=services_ids,
            to_delete=to_delete,
            to_create=to_create,
        )

    if to_create:
        ServiceResourceCounter.objects.bulk_create(to_create)
    if to_delete:
        ServiceResourceCounter.objects.filter(pk__in=to_delete).delete()

    log.info('Finish updating ServiceResourceCounter')


def _update_serviceresources_count_by_type(type_id, services_ids, to_delete, to_create):
    service_resources = (
        ServiceResource.objects
            .filter(
                service_id__in=services_ids,
                type_id=type_id,
                state__in=ServiceResource.ALIVE_STATES,
            )
            .values_list('service_id')
            .order_by()
            .annotate(count=Count('id'))
    )

    resource_map = {
        service_id: count
        for service_id, count in service_resources
    }

    counters = ServiceResourceCounter.objects.filter(
        service_id__in=services_ids,
        resource_type_id=type_id,
    )

    counters_map = {}
    for counter in counters:
        key = counter.service_id
        if key not in resource_map:
            # если счетчик есть, но в текущих данных таких
            # ресурсов нет - его можно удалять
            to_delete.append(counter.id)
        else:
            counters_map[key] = counter

    for service_id, count in resource_map.items():
        current_value = counters_map.get(service_id)
        if current_value is None:
            to_create.append(
                ServiceResourceCounter(
                    service_id=service_id,
                    resource_type_id=type_id,
                    count=count,
                )
            )
        elif current_value == 0:
            to_delete.append(current_value.id)
        elif current_value.count != count:
            # TODO: перейти на bulk_update в django 2.2
            current_value.count = count
            current_value.save()


@lock_task
def add_owner_role_in_secrets_robots():
    """
    Добавляет роли OWNER для Паспарту во все секреты от имени Гермеса
    """
    resources_robots = Resource.objects.filter(
        type__code=settings.ROBOT_RESOURCE_TYPE_CODE
    )
    plugin = RobotsPlugin()

    error = set()
    for robot in resources_robots:
        try:
            plugin.add_role_in_secret(
                resource=robot,
                role='OWNER',
                from_=settings.ABC_ZOMBIK_LOGIN,
                to_=settings.ABC_ROBOTSECRETS_LOGIN,
            )
        except (SupplierError, ClientError):
            error.add(robot.external_id)

    if error:
        log.warning(f'Didn\'t add a role: {error}')


@lock_task
def create_resources_for_robots():
    """
    Таска, которая нужна только в тестинге.
    Создаёт ресурсы роботов, которых ещё нет.
    """

    if yenv.type == 'production':
        message = 'Нельзя создать ресурсы роботов не в тестинге'
        log.warning(message)
        raise RobotsNotInTestingError(message)

    resources_robots = Resource.objects.filter(
        type__code=settings.ROBOT_RESOURCE_TYPE_CODE,
        external_id=OuterRef('login')
    )

    # логины роботов, которые не имеют ресурсов
    robots = (
        Staff.objects
        .filter(is_robot=True, is_dismissed=False)
        .annotate(has_resource=Exists(resources_robots))
        .filter(has_resource=False)
        .values_list('login', flat=True)
    )

    robot_type = ResourceType.objects.get(code=settings.ROBOT_RESOURCE_TYPE_CODE)

    # в тестинге у всех будет одинаковый, ну и что,
    # главное, чтобы он был, чтобы можно было привязывать
    # при неободимости можно будет проапдетить на другой тестовый
    test_secret_id = 'sec-01dbnrqx0c5q811y9xn8fc6tes'
    for robot_login in robots:
        Resource.objects.create(
            type=robot_type,
            external_id=robot_login,
            name=robot_login,
            link='{}/{}'.format(settings.STAFF_URL, robot_login),
            attributes={
                'secret_id': {
                    'value': test_secret_id,
                    'type': 'link',
                    'url': settings.VAULT_SECRET_URL.format(secret_id=test_secret_id)
                }
            },
        )


@lock_task
def upload_gdpr_to_yt():
    gdpr_resources = ServiceResource.objects.filter(
        resource__type__code=settings.GDPR_RESOURCE_TYPE_CODE
    ).granted().select_related('service', 'resource').order_by('id')
    items = []
    for gdpr_resource in gdpr_resources:
        destination: Service = gdpr_resource.service
        source_slug = gdpr_resource.resource.attributes.get('service_from_slug')
        attributes: Dict[str, Any] = gdpr_resource.resource.attributes
        if source_slug is None:
            log.error(f'GDPR resource {gdpr_resource.id} has no resource attribute `service_from_slug`')
            raise Exception('GDPR resource has no source service')

        source = Service.objects.get(slug=source_slug)
        items.append(
            {
                'destination_id': destination.id,
                'destination_name': destination.name,
                'source_id': source.id,
                'source_name': source.name,
                'data': attributes.get('data', ''),
                'use_data': attributes.get('is_using'),
                'store_data': attributes.get('is_storing'),
                'data_owner': attributes.get('subject'),
                'period_of_storage': attributes.get('store_for'),
                'storage_location': attributes.get('store_in'),
                'purpose_of_processing': attributes.get('purpose'),
                'path_to_data': attributes.get('path_to_data'),
                'access_from_yql_sql': attributes.get('access_from_yql_sql'),
                'access_from_api': attributes.get('access_from_api'),
                'access_from_web_cloud': attributes.get('access_from_web_cloud'),
                'access_from_file_based': attributes.get('access_from_file_based'),
            }
        )
    if items:
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)

        columns = list(items[0])
        query = [
            'USE hahn;',
            'INSERT INTO',
            f'`home/abc/gdpr/{datetime.date.today().isoformat()}`',
            f'({",".join(columns)})',
            'VALUES'
        ]
        value_rows = []
        for item in items:
            value_row = []
            for column in columns:
                value = item[column]
                if isinstance(value, int):
                    value_row.append(str(value))
                elif isinstance(value, str):
                    value = value.replace('"', '')
                    value_row.append(f'"{value}"')
                else:
                    value_row.append(f'"{value}"')
            value_rows.append(f'({", ".join(value_row)})')
        query.append(', '.join(value_rows))
        request = yql_client.query(' '.join(query))
        request.run()

        while request.status not in ('COMPLETED', 'ERROR'):
            log.debug(f'Export GDPR resources Yql request in status {request.status}')
            time.sleep(1.0)
        if request.status == 'ERROR':
            errors = '; '.join([error.format_issue() for error in request.errors])
            log.error(f'Error occurred on GDPR resources export: [{errors}]')
            raise Exception('Upload GDPR resources failed')


@lock_task
def sync_with_alert_provider():
    from yql.api.v1.client import YqlClient

    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)

    query = [
        'USE hahn;',
        'SELECT abc_slug, resource_id, service_provider_id, resource_type, responsible, environment, monitoring_stats ',
        'FROM `home/solomon/service_provider_alerts/service_provider_exports/all_resources`',
    ]

    request = yql_client.query(' '.join(query))
    request.run()

    resources_map = {}
    resource_codes = set()
    services = set()
    target_hash = set()

    for table in request.get_results():
        table.fetch_full_data()
        for row in table.rows:
            (
                abc_slug, resource_id, service_provider_id,
                resource_type, responsible, environment, monitoring_stats
            ) = row
            resource_code = f'{service_provider_id}_{resource_type}'
            hash_key = get_hash(*row)
            target_hash.add(hash_key)
            resources_map[hash_key] = {
                'abc_slug': abc_slug,
                'resource_id': resource_id,
                'resource_code': resource_code,
                'service_provider_id': service_provider_id,
                'resource_type': resource_type,
                'environment': environment,
                'monitoring_stats': monitoring_stats,
                'responsible': responsible,
            }
            resource_codes.add(resource_code)
            services.add(abc_slug)

    # получим данные из базы и сделаем агрегацию
    robot = get_abc_zombik()
    tags_map = {
        tag.slug.lower(): tag for tag in
        ResourceTag.objects.filter(
            category__slug='environment',
            type=ResourceTag.INTERNAL,
            service_id__isnull=True,
        )
    }

    resource_types = {
        rtype.code: rtype.id for rtype in
        ResourceType.objects.filter(code__in=resource_codes)
    }

    services = {
        service.slug: service.id for service in
        Service.objects.filter(slug__in=services)
    }

    current_resources = ServiceResource.objects.select_related('resource').filter(
        resource__provider_hash__isnull=False,
        type__category__slug='provider_alerts',
    ).all()
    current_resources_map = {}
    current_hash = set()
    for service_resource in current_resources:
        current_hash.add(service_resource.resource.provider_hash)
        key = get_hash(
            service_resource.type_id,
            service_resource.service_id,
            service_resource.resource.name,
        )
        current_resources_map[key] = service_resource

    # начнем собственно сверку и изменение/создание данных
    for hash_key, data in resources_map.items():
        if hash_key in current_hash:
            # уже есть такой ресурс
            continue

        # тут два варианта - либо такого нет, либо поменялось что-то
        resource_type_id = resource_types.get(data['resource_code'])
        service_id = services.get(data['abc_slug'])
        name = ','.join(
            [
                f'{key}:{value}'
                for key, value in
                data['resource_id'].items()
            ]
        )

        if not resource_type_id:
            log.warning(f'No resource type with code {data["resource_code"]} found')
            continue

        if not service_id:
            log.warning(f'No service with slug {data["abc_slug"]} found')
            continue

        service_resource = current_resources_map.get(
            get_hash(resource_type_id, service_id, name)
        )

        if not service_resource:
            log.info(f'Creating resource with hash: {hash_key}, name: {name}')

            resource = Resource.objects.create(
                type_id=resource_type_id,
                name=name,
                provider_hash=hash_key,
                attributes={
                    'resource_id': str(data['resource_id']),
                    'service_provider_id': data['service_provider_id'],
                    'resource_type': data['resource_type'],
                }
            )
            service_resource = ServiceResource.objects.create(
                type_id=resource_type_id,
                state=ServiceResource.GRANTED,
                service_id=service_id,
                granter=robot,
                granted_at=timezone.now(),
                resource_id=resource.id,
            )
            created = True

        else:
            # ресурс есть, но изменился
            created = False
            resource = service_resource.resource
            if resource.provider_hash == hash_key:
                # не должно такого быть
                log.error(f'Unexpected hash match: {service_resource.id}, {hash_key}')
            resource.provider_hash = hash_key

        update_fields_resource = {'provider_hash', }
        update_fields = []

        # поменям статус на выданный если нужно
        if service_resource.state != ServiceResource.GRANTED:
            service_resource.state = ServiceResource.GRANTED
            update_fields.append('state')

        # могли поменяться поля monitoring_stats / responsible / environment (теги)
        monitoring_stats = data['monitoring_stats']
        if resource.attributes.get('monitoring_stats', '') != str(monitoring_stats):
            resource.attributes['monitoring_stats'] = str(monitoring_stats)
            update_fields_resource.add('attributes')

            target_has_monitoring = all(
                value == 'true'
                for value in monitoring_stats['alerts_status'].values()
            )
            if service_resource.has_monitoring != target_has_monitoring:
                service_resource.has_monitoring = target_has_monitoring
                update_fields.append('has_monitoring')

        # обновим ответственного если нужно
        if resource.attributes.get('responsible') != data['responsible']:
            resource.attributes['responsible'] = data['responsible']
            update_fields_resource.add('attributes')

        # обновим теги если нужно
        if created or resource.type.has_editable_tags is False:
            target_tag_code = data['environment'].lower()
            service_tags = service_resource.tags.filter(
                category__slug='environment',
                type=ResourceTag.INTERNAL,
                service_id__isnull=True,
            )
            to_remove_tags = []
            has_target_tag = False
            for tag in service_tags:
                if tag.slug.lower() != target_tag_code:
                    to_remove_tags.append(tag)
                else:
                    has_target_tag = True

            if target_tag_code in tags_map and not has_target_tag:
                service_resource.tags.add(tags_map[target_tag_code])
            if to_remove_tags:
                service_resource.tags.remove(*to_remove_tags)

        # сохраним изменения
        if update_fields:
            service_resource.save(update_fields=update_fields)
        resource.save(update_fields=update_fields_resource)

    # удалим те которые пропали
    Resource.objects.filter(
        provider_hash__isnull=False,
        type__category__slug='provider_alerts',
    ).exclude(
        provider_hash__in=target_hash
    ).delete()


@lock_task
def sync_with_money_map(resource_code=None):
    from yql.api.v1.client import YqlClient

    yql_client = YqlClient(token=settings.YQL_OAUTH_TOKEN)
    resource_codes = [resource_code] if resource_code else settings.MONEY_MAP_RESOURCE_CODES
    money_map_tag = ServiceTag.objects.get(slug=settings.MONEY_MAP_TAG)
    robot = get_abc_zombik()
    all_services_with_resources = set()
    to_deprive = []
    for resource_code in resource_codes:
        query = [
            'USE hahn;',
            'SELECT abc_id, attributes, link ',
            f'FROM `//home/statkey/moneymap/abc_resource/{resource_code}`',
        ]

        request = yql_client.query(' '.join(query))
        request.run()

        target_resources = {}
        for table in request.get_results():
            table.fetch_full_data()
            for row in table.rows:
                abc_id, attributes, link = row
                all_services_with_resources.add(abc_id)
                target_resources[abc_id] = {
                    'abc_id': abc_id,
                    'attributes': yson_to_json(attributes),
                    'link': link,
                }

        if not target_resources:
            log.error(f'Got blank response for : {resource_code}')
            raise ResourceImporterError

        resource_type = ResourceType.objects.get(code=resource_code)
        current_resources = ServiceResource.objects.filter(
            type=resource_type,
        ).select_related('resource', 'service').prefetch_related('service__tags')
        current_resources_map = {}
        for service_resource in current_resources:
            if service_resource.service_id not in target_resources:
                to_deprive.append(service_resource)
            else:
                current_resources_map[service_resource.service_id] = service_resource

        existing_services = {obj.id for obj in Service.objects.filter(pk__in=target_resources)}

        # начнем собственно сверку и изменение/создание данных
        for service_id, data in target_resources.items():
            if service_id not in existing_services:
                log.warning(f'Skipping not existing service: {service_id}')
                continue

            if service_id not in current_resources_map:
                log.info(f'Creating resource for {service_id}, {resource_code}')
                resource = Resource.objects.create(
                    type=resource_type,
                    name=f'moneymap-{resource_code}',
                    link=data['link'],
                    attributes={
                        'attributes': data['attributes'],
                    }
                )
                service_resource = ServiceResource.objects.create(
                    type=resource_type,
                    state=ServiceResource.GRANTED,
                    service_id=service_id,
                    granter=robot,
                    granted_at=timezone.now(),
                    resource_id=resource.id,
                )
                service = Service.objects.get(pk=service_id)
                service.tags.add(money_map_tag)
            else:
                # ресурс есть, нужно проверить не изменился ли
                service_resource = current_resources_map[service_id]
                if money_map_tag not in service_resource.service.tags.all():
                    service_resource.service.tags.add(money_map_tag)

            # поменям статус на выданный если нужно
            if service_resource.state != ServiceResource.GRANTED:
                service_resource.state = ServiceResource.GRANTED
                service_resource.save(update_fields=('state',))

            update_fields = []
            if service_resource.resource.attributes != data['attributes']:
                service_resource.resource.attributes = {
                    'attributes': data['attributes'],
                }
                update_fields.append('attributes')

            if service_resource.resource.link != data['link']:
                service_resource.resource.link = data['link']
                update_fields.append('link')

            # сохраним изменения
            if update_fields:
                service_resource.resource.save(update_fields=update_fields)

    # удалим те которые пропали
    if to_deprive:
        ServiceResource.objects.filter(
            pk__in=(obj.id for obj in to_deprive)
        ).forced_deprive()

    # уберем тег там, где флагов не осталось
    if all_services_with_resources:
        for service in Service.objects.exclude(
            pk__in=all_services_with_resources,
        ).filter(tags=money_map_tag):
            service.tags.remove(money_map_tag)
