from django.contrib import admin

from plan.api.idm.actions import delete_role
from plan.common.base_admin import ViewModelAdmin
from plan.roles.models import Role, RoleScope
from plan.services.models import ServiceMember, ServiceMemberDepartment


class IsGlobalListFilter(admin.SimpleListFilter):
    title = 'Специфичность'

    parameter_name = 'specificity'

    def lookups(self, request, model_admin):
        return (
            ('global', 'Глобальная'),
            ('service', 'Конкретного сервиса'),
        )

    def queryset(self, request, queryset):
        qs = queryset
        if self.value() == 'global':
            qs = queryset.globalwide()
        elif self.value() == 'service':
            qs = queryset.service_specific()
        return qs


class IsUtilityListFilter(admin.SimpleListFilter):
    title = 'Тип скоупа'

    parameter_name = 'scope_utility'

    def lookups(self, request, model_admin):
        return (
            ('utility_scope', 'Служебный'),
            ('functional_scope', 'Функциональный'),
        )

    def queryset(self, request, queryset):
        qs = queryset
        if self.value():
            value = self.value() == 'utility_scope'
            qs = queryset.filter(scope__utility_scope=value)
        return qs


class IsProtectedListFilter(admin.SimpleListFilter):
    title = 'Защищенный скоуп'

    parameter_name = 'scope_protected'

    def lookups(self, request, model_admin):
        return (
            ('yes', 'Да'),
            ('no', 'Нет'),
        )

    def queryset(self, request, queryset):
        qs = queryset
        if self.value():
            value = self.value() == 'yes'
            qs = queryset.filter(scope__protected=value)
        return qs


@admin.register(RoleScope)
class RoleScopeAdmin(ViewModelAdmin):
    list_display = ('slug', 'name', 'protected', 'utility_scope', )


@admin.register(Role)
class RoleAdmin(ViewModelAdmin):
    list_display = (
        'name', 'name_en', 'native_lang',
        'scope', 'weight', 'service',
        'get_protected', 'get_utility_scope',
    )
    list_filter = (
        'scope', IsGlobalListFilter,
        IsUtilityListFilter, IsProtectedListFilter,
    )
    search_fields = (
        'name', 'name_en', 'scope__name', 'scope__name_en', 'service__name', 'service__name_en', 'service__slug'
    )

    actions = ['drop']

    def get_utility_scope(self, obj):
        return obj.scope.utility_scope
    get_utility_scope.short_description = 'Служебный скоуп'

    def get_protected(self, obj):
        return obj.scope.protected
    get_protected.short_description = 'Защищенный'

    def drop(self, request, queryset):
        for role in queryset:
            if not role.service:
                self.message_user(request, 'Невозможно удалить общую роль %s' % role)
                continue

            if ServiceMember.objects.filter(role=role).exists() \
                    or ServiceMemberDepartment.objects.filter(role=role).exists():
                self.message_user(request, 'Невозможно удалить роль %s: у нее есть участники' % role)
                continue

            delete_role(role)
            role.delete()
    drop.short_description = 'Удалить роль в IDM'
