import logging

from django.conf import settings
from django.db.models import Q
from django_filters import CharFilter
from rest_framework import serializers, viewsets
from rest_framework.response import Response

from plan.api import base
from plan.api.mixins import TvmAccessMixin
from plan.api.exceptions import ValidationError
from plan.api.fields import MappingField
from plan.api.filters import CustomModelMultipleChoiceFilter, PlanFilterSet
from plan.api.idm import actions
from plan.api.mixins import DefaultFieldsMixin
from plan.api.permissions import TvmAuthenticated
from plan.api.validators import UniqueTogetherLookupValidator
from plan.roles.api.rolescopes import RoleScopeSerializer
from plan.roles.api.validators import scope_is_not_protected, RoleNameIsUniqueWithinScope
from plan.roles.models import Role, RoleScope
from plan.services.models import Service
from plan.swagger import SwaggerServices

log = logging.getLogger(__name__)


class RoleSerializer(base.ModelSerializer):
    service = base.CompactServiceSerializer()
    scope = RoleScopeSerializer()
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = Role
        fields = ('id', 'name', 'service', 'scope', 'code', 'created_at')


class CreateRoleSerializer(base.ModelSerializer):
    service = serializers.PrimaryKeyRelatedField(
        queryset=Service.objects.all(),
        required=True,
    )
    scope = serializers.PrimaryKeyRelatedField(
        queryset=RoleScope.objects.all(),
        required=True,
        validators=[scope_is_not_protected]
    )
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    id = serializers.IntegerField(read_only=True)
    fields_mapping_ = {'name': ('name', 'name_en')}
    native_lang = serializers.CharField(default=settings.DEFAULT_NATIVE_LANGUAGE)

    class Meta:
        model = Role
        fields = ('name', 'service', 'scope', 'code', 'id', 'native_lang')

        validators = [
            UniqueTogetherLookupValidator(
                queryset=Role.objects.all(),
                fields_with_lookup=('service', 'name__iexact'),
                message='В этом сервисе уже есть роль с таким именем.',
            ),
            UniqueTogetherLookupValidator(
                queryset=Role.objects.all(),
                fields_with_lookup=('service', 'name_en__iexact'),
                message='В этом сервисе уже есть роль с таким английским именем.',
            ),
            RoleNameIsUniqueWithinScope()
        ]


class RoleFilter(PlanFilterSet):
    service = base.RoleIncludeGlobalFilter(
        field_name='service',
        queryset=Service.objects.all(),
    )
    service__slug = base.RoleIncludeGlobalFilter(
        field_name='service__slug',
        queryset=Service.objects.all(),
        to_field_name='slug',
    )
    service__with_descendants = base.ServiceWithDescendantsFilter(
        field_name='service',
        queryset=Service.objects.all(),
        distinct=False,
        only_active=True,
    )
    service__with_descendants__slug = base.ServiceWithDescendantsFilter(
        field_name='service__slug',
        queryset=Service.objects.all(),
        to_field_name='slug',
        distinct=False,
        only_active=True,
    )
    scope = CustomModelMultipleChoiceFilter(
        field_name='scope',
        queryset=RoleScope.objects.all(),
    )
    is_exportable = base.BooleanFilter(lookup_expr='exact')
    scope__can_issue_at_duty_time = base.BooleanFilter(method='_can_issue_at_duty_time')
    can_issue_at_duty_time = base.BooleanFilter(method='_can_issue_at_duty_time')
    available_for = CharFilter(method='_available_for')

    class Meta:
        model = Role
        fields = {
            'id': ['exact', 'in'],
            'code': ['exact', 'contains', 'in'],
            'name': ['exact', 'contains'],
            'name_en': ['exact', 'contains'],
            'scope': ['exact', 'in'],
            'service': ['exact', 'in'],
            'service__slug': ['exact', 'in'],
            'is_exportable': ['exact'],
        }

    def _can_issue_at_duty_time(self, queryset, name, value):
        if value in ('1', 'True', 'true'):
            queryset = queryset.exclude(
                code__in=Role.CAN_NOT_USE_FOR_DUTY
            ).exclude(
                scope__can_issue_at_duty_time=False,
            )
        return queryset

    def _available_for(self, queryset, name, value):
        if value:
            if not value.isdigit():
                service = Service.objects.filter(slug=value).first()
                value = getattr(service, 'id', None)
            if value:
                queryset = queryset.filter(
                    Q(service_id=value) | Q(service_id=None)
                )
        return queryset


class RolesView(TvmAccessMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """
    Роли
    """
    permission_classes = [TvmAuthenticated]
    serializer_class = RoleSerializer
    http_method_names = ['get', 'post']
    queryset = Role.objects.all().select_related('service', 'scope').order_by('pk')
    filter_class = RoleFilter

    def update(self, request, *args, **kwargs):
        return

    def create(self, request):
        """
        Создание кастомной роли в сервисе.
        Кастомная роль доступна только внутри одного конкретного сервиса.
        """
        serializer = CreateRoleSerializer(data=request.data)
        if serializer.is_valid():
            serializer.save()
            role = serializer.instance

            log.info('Add custom role %r by %s', serializer.instance, self.request.person.login)
            actions.add_role(role)
            return Response(RoleSerializer(role).data)
        else:
            raise ValidationError(extra=serializer.errors)


class V4RolesView(DefaultFieldsMixin, RolesView):
    default_swagger_schema = SwaggerServices

    pagination_class = base.ABCCursorPagination
    ordering_fields = ('id',)

    default_fields = [
        'code',
        'id',

        'scope.id',
        'scope.slug',
    ]
