import logging

from rest_framework import viewsets

from plan.api import base
from plan.api.fields import MappingField
from plan.api.filters import PlanFilterSet
from plan.api.mixins import DefaultFieldsMixin, TvmAccessMixin
from plan.api.permissions import TvmAuthenticated
from plan.roles.models import RoleScope
from plan.swagger import SwaggerServices


log = logging.getLogger(__name__)


class RoleScopeSerializer(base.ModelSerializer):
    name = MappingField({'ru': 'name', 'en': 'name_en'})
    fields_mapping_ = {'name': ('name', 'name_en')}

    class Meta:
        model = RoleScope
        fields = (
            'id', 'slug', 'name',
            'protected', 'utility_scope',
        )


class RoleScopeFilter(PlanFilterSet):

    class Meta:
        model = RoleScope
        fields = {
            'id': ['exact', 'in'],
            'protected': ['exact'],
        }


class RoleScopesView(TvmAccessMixin, base.OrderingMixin, viewsets.ModelViewSet):
    """
    Скоупы ролей
    """
    _permissions_to_proceed = 'view_own_services'
    permission_classes = [TvmAuthenticated]
    serializer_class = RoleScopeSerializer
    filter_class = RoleScopeFilter
    search_fields = ('slug', 'name', 'name_en')  # TODO: ?
    http_method_names = ['get']
    queryset = RoleScope.objects.all().order_by('pk')


class V4RoleScopesView(DefaultFieldsMixin, RoleScopesView):
    default_swagger_schema = SwaggerServices

    pagination_class = base.ABCCursorPagination
    ordering_fields = ('id',)

    default_fields = ['id', 'slug']
